/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:25:51 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.symreg;

import gpsys.*;
import gpsys.primitives.*;

/**
 * The Fitness class for the symbolic regression problem.  The fitness is
 * measured using two values.  The first is the error between the real
 * function and the function generated by the GP system.  The second value is
 * the complexity of the GP program.  The fitness of one GP is is deemed better
 * than another if either the error is smaller or if the error is equal, but
 * the complexity is smaller.  The termination criteria is met when the error
 * is zero.  This problem involves finding the function :-
 * <pre>
 *          x^3 + 2x^2 +3x + 5
 * </pre>
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:25:51 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class SymRegFitness extends Fitness {
    /**
     * The error between the ideal function and the GP.
     */
    double  fitness;
    /**
     * The complexity of the GP i.e. the number of Genes it contains.
     */
    int     complexity;

    /**
     * The minimum value of x to be used for testing the evolved function.
     */
    public static float from    = 0.0f;;
    /**
     * The maximum value of x to be used for testing the evolved function.
     */
    public static float to      = 100.0f;
    /**
     * The number of samples used test the evolved function.
     */
    public static int samples   = 100;

    /**
     * Construct a Fitness object with default Fitness.
     */
    public SymRegFitness() {
        fitness = 0.0f;
        complexity  = 0;
    }

    /**
     * Constructs a Fitness object by evaluating an Individual.
     *
     * @param   gpParameters    The GP parameters for this run.
     * @param   i               The individual to be evaluated.
     */
    public SymRegFitness(GPParameters gpParameters, Individual i) {

        complexity = i.complexity();

        // calculate the raw fitness

        fitness = 0.0;
        float step = (to - from) / samples;

        // xFloat is reference to the XFloat Terminal used to generate ADF0.
        XFloat xFloat = (XFloat) gpParameters.adf[0].terminals[0];

        for (float j = from; j <= to; j += step) {
            xFloat.set(j);
            float guess = 0.0f;
            float real  = j*j*j + 2.0f*j*j +3.0f*j + 5.0f;

            try {
                guess = i.evaluateFloat();
                fitness += Math.abs(real - guess);
            }
            catch (DivideByZeroException e) {
                // a divide by zero error is really bad
                // so don't bother with the remaining tests
                fitness = Float.MAX_VALUE;
                break;

            }
            catch (GPException e) {
                gpParameters.observer.exception(e);
            }
        }
    }

    /**
     * Adds a fitness value to this fitness value.
     *
     * @param f the fitness value to be added.
     */
    public void add(Fitness f) {
        fitness += ((SymRegFitness)f).fitness;
        complexity  += ((SymRegFitness)f).complexity;
    }

    /**
     * Divide the fitness by the specified integer.  This is used by the
     * GP system to calculate the average fitness of the population.
     *
     * @param   divisor the integer to divide the fitness by.
     */
    public void divide(int divisor) {
        fitness /= divisor;
        complexity  /= divisor;
    }

    /**
     * Tests if this fitness value is greater than another fitness value.
     *
     * @param   f   the fitness with which to compare.
     * @return  true if f has higher fitness, false otherwise.
     */
    public boolean greaterThan(Fitness f) {
        if (fitness < ((SymRegFitness)f).fitness)
            return true;
        if (fitness == ((SymRegFitness)f).fitness)
            return complexity  < ((SymRegFitness)f).complexity;
        return false;
    }

    /**
     * Tests if this fitness is less than another fitness.
     *
     * @param   f   the fitness with which to compare.
     * @return  true if f has less fitness, false otherwise.
     */
    public boolean lessThan(Fitness f) {
        if (fitness > ((SymRegFitness)f).fitness)
            return true;
        if (fitness == ((SymRegFitness)f).fitness)
            return complexity  > ((SymRegFitness)f).complexity;
        return false;
    }

    /**
     * Tests if this fitness is equal to another fitness.
     *
     * @param   f   the fitness with which to compare.
     * @return  true if f has the same fitness, false otherwise.
     */
    public boolean equals(Fitness f) {
        return
            (fitness == ((SymRegFitness)f).fitness) &&
            (complexity  == ((SymRegFitness)f).complexity);
    }

    /**
     * Creates a new instance of the Fitness object with a default fitness.
     *
     * @return  an instance of a Fitness object with default fitness.
     */
    public Fitness instance() {
        return new SymRegFitness();
    }

    /**
     * Creates a new instance of the Fitness object which represents the
     * fitness of the specified individual.
     *
     * @param   gpParameters    the parameters for this GP run.
     * @param   i               the individual to be evaluated.
     * @return  The Fitness of the specified individual.
     */
    public Fitness instance(GPParameters gpParameters, Individual i) {
        return new SymRegFitness(gpParameters, i);
    }

    /**
     * Tests whether this fitness meets the termination criteria.
     *
     * @return  true if the termination criteria has been met, false otherwise.
     */
    public boolean terminationCondition() {
        return fitness == 0;
    }

    /**
     * Converts the fitness into a String suitable for printing.
     *
     * @return  A String representing the fitness.
     */
    public String toString() {
        return "Fitness(" + fitness + "," + complexity + ")";
    }
}
