/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:25:51 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.symreg;

import gpsys.*;
import java.io.*;

/**
 * The user interface for the symbolic regression problem.  This class
 * also contains the main() function to execute the application.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:25:51 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class SymReg implements GPObserver {
    /**
     * The filePrefix to use when saving reports and generation states during
     * evolution.
     */
    String filePrefix;

    /**
     * Construct the symbolic regression user interface using the specified
     * file prefix.
     *
     * @param filePrefix    The file prefix to bse used for saving reports and
     *                      generation states.
     */
    public SymReg(String filePrefix) {
        this.filePrefix = filePrefix;
    }

    /**
     * If the filePrefix is null, just write a report of the current generation
     * to the standard output.  Otherwise, also append the report to the file
     * "filePrefix.txt" and save the current generation to the file
     * "filePrefix.p1.gzip".
     *
     * @param gpParameters  The GP parameters used for this run.
     * @param how           How the generation was created.  Can be either
     *                      CREATION, FROMSTREAM or EVOLVED.
     */
    public void generationUpdate(GPParameters gpParameters, int how) {
        // if a file prefix was given, write the report to file and save the
        // current generation.  This need not be done if the generation was
        // just loaded from file.
        if ((filePrefix != null) &&
            (how == GPObserver.CREATION || how == GPObserver.EVOLVED)) {
            // try to save the current generation
            try {
                diagnosticUpdate("Saving current generation...");
                gpParameters.save(filePrefix);
                diagnosticUpdate("Saved  current generation.");
            }
            catch (IOException e) {
                System.out.println("gpParameters.save() : " + e);
                System.exit(1);
            }

            // try to write a report of the current population to file
            try {
                gpParameters.writeReport(filePrefix,how == GPObserver.CREATION);
            }
            catch (IOException e) {
                System.out.println("gpParameters.writeReport() : " + e);
                System.exit(1);
            }
        }

        // now write a report of the current population to the standard output
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        gpParameters.writeReport(pw,
            (how == GPObserver.CREATION) || (how == GPObserver.FROMSTREAM));
        System.out.print(sw.toString());

        // if the termination criteria has been reached then stop.
        if (gpParameters.population.bestRun.fitness.terminationCondition()) {
            System.out.println("GP run completed..");
            System.exit(0);
        }
    }

    /**
     * We are not interested in this update, so we just ignore it.
     *
     * @param   gpParameters    the GP parameters used for this run.
     * @param   i               the Individual that has just been created.
     * @param   creationIndex   the index of the Individual in the population.
     * 
     */
    public void individualUpdate(GPParameters gpParameters,
        Individual i, int creationIndex) {
    }

    /**
     * We are not interested in this update, so we just ignore it.
     *
     * @param   gpParameters    the GP parameters used for this run.
     * @param   individualIndex the index of the created Individual in the
     *                          population.
     * @param   creationMethod  how the Individual was created.  Can be either
     *                          VIA_MUTATION or VIA_CROSSOVER.
     */
    public void individualUpdate(GPParameters gpParameters,
        int individualIndex, int creationIndex) {
    }

    /**
     * Print the diagnostic message to the standard output.
     *
     * @param s The diagnostic message.
     */
    public void diagnosticUpdate(String s) {
        System.out.println(s);
    }

    /**
     * Print the exception and generate a stack trace on the standard output.
     *
     * @param e The exception that was genearted.
     */
    public void exception(GPException e) {
        System.out.println(e.getMessage());
        e.printStackTrace();
        System.exit(0);
    }

    /**
     * The main() method of the symbolic regression application.  This
     * application may be invoked in any of the following ways.
     * <pre>
     *      java SymReg &lt;filePrefix&gt;
     *      java SymReg &lt;filePrefix&gt; &lt;generations>
     *      java SymReg &lt;filePrefix&gt; &lt;RNG seed> &lt;population> &lt;generations>
     *      java SymReg &lt;RNG seed&gt; &lt;population> &lt;generations>
     * </pre>
     * 
     * The first istructs the application to restart from the last saved
     * session using the files with the prefix specified.  The second is the
     * same as the first, execept that the maximum number of geneartion is
     * modified as specified.  The third istructs the application to start
     * a new run using the specifed file prefix for saves, and the
     * specified parameters for the run.  The last is the similar to the
     * previous invokation except that nothing is saved to disk and is hence
     * very fast.
     */
    public static void main(String[] argv) {

        // change this to whatever is suitable to you!!
        java.util.Locale.setDefault(java.util.Locale.UK);

        SymRegGPParameters gpParameters = null;
        String filePrefix = null;

        // now read the command line arguments
        switch(argv.length) {
            case 1: // load entirely from saved session
            case 2: // load from saved session but change the max generations
                filePrefix = argv[0];
                try {
                    System.out.println("Loading last saved generation...");
                    gpParameters =
                        (SymRegGPParameters) GPParameters.load(filePrefix);
                    if (gpParameters != null)
                        System.out.println("Loaded  last saved generation...");
                }
                catch (java.io.IOException e) {
                    System.out.println("Loading problem : " + e);
                    System.exit(1);
                }
                catch (ClassNotFoundException e) {
                    System.out.println("Loading problem : " + e);
                    System.exit(1);
                }
                if (argv.length == 2) {
                    int generations = Integer.parseInt(argv[1]);
                    gpParameters.generations = generations;
                }
                break;
            case 3: {
                    long rngSeed    = Long.parseLong(argv[0]);
                    int population  = Integer.parseInt(argv[1]);
                    int generations = Integer.parseInt(argv[2]);
                    gpParameters = new
                        SymRegGPParameters(rngSeed, population, generations);
                }
                break;
            case 4: {
                    filePrefix = argv[0];
                    long rngSeed    = Long.parseLong(argv[1]);
                    int population  = Integer.parseInt(argv[2]);
                    int generations = Integer.parseInt(argv[3]);
                    gpParameters = new
                        SymRegGPParameters(rngSeed, population, generations);
                }
                break;
            default:
                System.out.println("Usage : SymReg <file>");
                System.out.println("        SymReg <file> <generations>");
                System.out.println("        SymReg " +
                    "<RNG seed> <population> <generations>");
                System.out.println("        SymReg " +
                    "<file> <RNG seed> <population> <generations>");
                return;
        }

        // set the observer to be an instance of our user interface
        gpParameters.observer = new SymReg(filePrefix);

        // create a new GP system
        GPsys gpSys = new GPsys(gpParameters);
        // and start evolving !!!
        gpSys.evolve();
    }
}
