/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:24:33 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.primitives;

import gpsys.*;

/**
 * <pre>
 *  &lt;typeX&gt; ProgN(&lt;typeX&gt; a1, &lt;typeX&gt; a2, ..., &lt;typeX&gt; aN)
 * </pre>
 *
 * ProgN is a generic function that takes N arguments of the same type,
 * evaluates all of them, and returns the value of the last argument.  Supported
 * Types include OBJECT, BYTE, SHORT, INT, LONG, FLOAT, DOUBLE, CHAR and
 * BOOLEAN.
 *
 * @see         gpsys.Type
 * @see         gpsys.Primitive
 * @see         gpsys.Function
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:24:33 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class ProgN extends Function {

    /**
     * Constructs a new ProgN Function which takes the specified number of
     * arguments, all of which are of the specified type.
     *
     * @param returnType    The type of all the arguments.  This must be one
     *                      one of OBJECT, BYTE, SHORT, INT, LONG, FLOAT,
     *                      DOUBLE, CHAR or BOOLEAN.
     * @param argNum        The number of arguments taken by the Function.
     */
    public ProgN(Type returnType, int argNum) {
        this.type = returnType;
        argTypes = new Type[argNum];
        for (int i = 0; i < argNum; i++)
            argTypes[i] = returnType;
    }

    /**
     * Used to evaluate a ProgN Function of Type OBJECT.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return An Object reference, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final Object evaluateObject(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateObject(i);
        return arguments[argTypes.length - 1].evaluateObject(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type BYTE.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A byte, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final byte evaluateByte(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateByte(i);
        return arguments[argTypes.length - 1].evaluateByte(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type SHORT.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A short, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final short evaluateShort(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateShort(i);
        return arguments[argTypes.length - 1].evaluateShort(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type INT.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return An int, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final int evaluateInt(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateInt(i);
        return arguments[argTypes.length - 1].evaluateInt(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type LONG.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A long, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final long evaluateLong(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateLong(i);
        return arguments[argTypes.length - 1].evaluateLong(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type FLOAT.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A float, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final float evaluateFloat(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateFloat(i);
        return arguments[argTypes.length - 1].evaluateFloat(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type DOUBLE.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A double, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final double evaluateDouble(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateDouble(i);
        return arguments[argTypes.length - 1].evaluateDouble(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type CHAR.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A char, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final char evaluateChar(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateChar(i);
        return arguments[argTypes.length - 1].evaluateChar(i);
    }

    /**
     * Used to evaluate a ProgN Function of Type BOOLEAN.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees representing the arguments to be
     *                  evaluated.
     * @return A boolean, which is the value of the last argument.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final boolean evaluateBoolean(Individual i, Gene[] arguments)
    throws EvaluationException {
        for (int arg = 0; arg < argTypes.length - 1; arg++)
            arguments[arg].evaluateBoolean(i);
        return arguments[argTypes.length - 1].evaluateBoolean(i);
    }

    /**
    * Creates an instance of this ProgN object.  The Type instantiation is
    * preserved so that if an INT ProgN is being cloned, the clone will also be
    * INT instantiated.  So to is the number of arguments to expected (the
    * value of N).
    *
    * @return   A reference to this Object (yes the same object, since
    *           the type and argument information is to be the same,
    *           there are therefore no instance variables to be modified.
    */
    public final Primitive instance() {
        return this;
    }

    /**
    * Returns a String representation of this Function.
    *
    * @return A String containing the name of the Function.
    */
    public String toString() {
        return "ProgN";
    }
}
