/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:24:33 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.primitives;

import gpsys.*;


/**
 * <pre>
 *  &lt;typeX&gt; Mul(&lt;typeX&gt; num1, &lt;typeX&gt; num2)
 * </pre>
 *
 * Mul is a generic function that returns the result of multiplying both
 * arguments.
 * Mul needs to be Type instantiated during construction to work with
 * particular argument Types.  The supported argument Types include BYTE,
 * SHORT, INT, LONG, FLOAT and DOUBLE.
 *
 * @see         gpsys.Type
 * @see         gpsys.Primitive
 * @see         gpsys.Function
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:24:33 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Mul extends Function {

    /**
    * Constructs a Mul Function that works with the specified Type.  The
    * supported Types include BYTE, SHORT, INT, LONG, FLOAT and DOUBLE.
    *
    * @param type   The Type of this Mul Function.
    */
    public Mul(Type type) {
        this.type = type;
        argTypes = new Type[2];
        argTypes[0] = type;
        argTypes[1] = type;
    }

    /**
    * Used to multiply arguments of BYTE Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be added.
    * @return A byte representing the product of the arguments.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final byte evaluateByte(Individual i, Gene[] arguments)
    throws EvaluationException {
        return (byte)
             (arguments[0].evaluateByte(i) * arguments[1].evaluateByte(i));
    }

    /**
    * Used to multiply arguments of SHORT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be added.
    * @return A short representing the product of the arguments.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final short evaluateShort(Individual i, Gene[] arguments)
    throws EvaluationException {
        return (short)
             (arguments[0].evaluateShort(i) * arguments[1].evaluateShort(i));
    }

    /**
    * Used to multiply arguments of INT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be added.
    * @return An int representing the product of the arguments.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final int evaluateInt(Individual i, Gene[] arguments)
    throws EvaluationException {
        return
             arguments[0].evaluateInt(i) *
             arguments[1].evaluateInt(i);
    }

    /**
    * Used to multiply arguments of LONG Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be added.
    * @return A long representing the product of the arguments.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final long evaluateLong(Individual i, Gene[] arguments)
    throws EvaluationException {
        return
             arguments[0].evaluateLong(i) *
             arguments[1].evaluateLong(i);
    }

    /**
    * Used to multiply arguments of FLOAT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be added.
    * @return A float representing the product of the arguments.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final float evaluateFloat(Individual i, Gene[] arguments)
    throws EvaluationException {
        return
             arguments[0].evaluateFloat(i) *
             arguments[1].evaluateFloat(i);
    }

    /**
    * Used to multiply arguments of DOUBLE Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be added.
    * @return A double representing the product of the arguments.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final double evaluateDouble(Individual i, Gene[] arguments)
    throws EvaluationException {
        return
             arguments[0].evaluateDouble(i) *
             arguments[1].evaluateDouble(i);
    }

    /**
    * Creates an instance of this Mul object.  The Type instantiation is
    * preserved so that if an INT Mul is being cloned, the clone will also be
    * INT instantiated.
    *
    * @return   A reference to this Object (yes the same object, since
    *           the type information is to be the same, there are therefore
    *           no instance variables to be modified.
    */
    public final Primitive instance() {
        return this;
    }

    /**
    * Returns a String representation of this Function.
    *
    * @return A String containing the name of the Function.
    */
    public String toString() {
        return "Mul";
    }
}
