/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:24:33 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.primitives;

import gpsys.*;

/**
 * <pre>
 *  &lt;typeX&gt; MemSet(MEM_typeX memory, INT index, &lt;typeX&gt; value)
 * </pre>
 *
 * MemSet is a generic Function which takes three arguments.  The first is an
 * indexed memory.  The second is an index into that memory.  The third is the
 * value that the specified location of the specified memory needs to be set.
 * MemGet needs to be instantiated to work with a particular type of memory.
 * Valid memory types include MEM_OBJECT, MEM_BYTE, MEM_SHORT, MEM_INT,
 * MEM_LONG, MEM_FLOAT, MEM_DOUBLE, MEM_CHAR, and MEM_BOOLEAN.
 *
 * @see         gpsys.Type
 * @see         gpsys.Primitive
 * @see         gpsys.Terminal
 * @see         gpsys.MemGet
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:24:33 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class MemSet extends Function {

    /**
     * Construct a new MemSet Function to assign values to the specified
     * memory type, which holds the specified type of elements.
     *
     * @param memoryType    The type of the memory, can be one of MEM_OBJECT,
     *                      MEM_BYTE, MEM_SHORT, MEM_INT,MEM_LONG, MEM_FLOAT,
     *                      MEM_DOUBLE, MEM_CHAR, and MEM_BOOLEAN.
     * @param returnType    The type that the memory holds, can be one of
     *                      OBJECT, BYTE, SHORT, INT, LONG, FLOAT, DOUBLE,
     *                      CHAR, and BOOLEAN.
     */
    public MemSet(Type memoryType, Type returnType) {
        this.type = returnType;
        argTypes = new Type[3];
        argTypes[0] = memoryType;
        argTypes[1] = Type.INT;
        argTypes[2] = returnType;
    }

    /**
     * Used to set values in memories of MEM_OBJECT type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  An Object reference representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final Object evaluateObject(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemObject mo = (MemObject) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateObject(i));
    }

    /**
     * Used to set values in memories of MEM_BYTE type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A byte representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final byte evaluateByte(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemByte mo = (MemByte) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateByte(i));
    }

    /**
     * Used to set values in memories of MEM_SHORT type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A short representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final short evaluateShort(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemShort mo = (MemShort) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateShort(i));
    }

    /**
     * Used to set values in memories of MEM_INT type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  An int representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final int evaluateInt(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemInt mo = (MemInt) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateInt(i));
    }

    /**
     * Used to set values in memories of MEM_LONG type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A long representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final long evaluateLong(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemLong mo = (MemLong) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateLong(i));
    }

    /**
     * Used to set values in memories of MEM_FLOAT type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A float representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final float evaluateFloat(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemFloat mo = (MemFloat) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateFloat(i));
    }

    /**
     * Used to set values in memories of MEM_DOUBLE type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A double representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final double evaluateDouble(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemDouble mo = (MemDouble) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateFloat(i));
    }

    /**
     * Used to set values in memories of MEM_CHAR type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A char representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final char evaluateChar(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemChar mo = (MemChar) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateChar(i));
    }

    /**
     * Used to set values in memories of MEM_BOOLEAN type.
     *
     * @param i         The individual being evaluated.
     * @param arguments The Gene trees supplied as arguments to the Function,
     *                  the first argument is the memory to be written to,
     *                  the second argument is the index of the memory location
     *                  to be assigned, and third is the value to be assigned.
     * @return  A boolean representing the new value of the memory
     *          at the specified location.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public final boolean evaluateBoolean(Individual i, Gene[] arguments)
    throws EvaluationException {
            MemBoolean mo = (MemBoolean) arguments[0].evaluateObject(i);
            return
                mo.set(
                    arguments[1].evaluateInt(i),
                    arguments[2].evaluateBoolean(i));
    }

    /**
    * Creates an instance of this MemSet object.  The Type instantiation is
    * preserved so that if an MEM_INT MemSet is being cloned, the clone will
    * also be MEM_INT instantiated.
    *
    * @return   A reference to this Object (yes the same object, since
    *           the type information is to be the same, there are therefore
    *           no instance variables to be modified.
    */
    public final Primitive instance() {
        return this;
    }

    /**
    * Returns a String representation of this Function.
    *
    * @return A String containing the name of the Function.
    */
    public String toString() {
        return "MemSet";
    }
}
