/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:24:33 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.primitives;

import gpsys.*;


/**
 * <pre>
 *  &lt;TypeX&gt; if (BOOLEAN condition, &lt;typeX&gt; trueValue, &lt;typeX&gt; falseValue)
 * </pre>
 *
 * If is a generic function that conditionally evaluates one of two arguments.
 * The function evaluates the first argument (which is of Type BOOLEAN).  If the
 * result is true, then the result of evaluating the second argument is
 * returned, otherwise the result of evaluating the third argument is
 * returned.  The second and third arguments must be of the same type.
 * If needs to be Type instantiated during construction to work with particular
 * argument Types.  The supported argument
 * Types include BYTE, SHORT, INT, LONG, FLOAT, DOUBLE, CHAR and BOOLEAN.
 *
 * @see         gpsys.Type
 * @see         gpsys.Primitive
 * @see         gpsys.Function
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:24:33 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class If extends Function {

    /**
     * Constructs an If Function that works with the specified Type.  The
     * supported Types include BYTE, SHORT, INT, LONG, FLOAT, DOUBLE, CHAR,
     * BOOLEAN, and OBJECT.
     *
     * @param type  The Type of this If Function.
     */
    public If(Type returnType) {
        type = returnType;
        argTypes = new Type[3];
        argTypes[0] = Type.BOOLEAN;
        argTypes[1] = returnType;
        argTypes[2] = returnType;
    }

    /**
    * Used to conditionally evaluate arguments of OBJECT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return An Object reference representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final Object evaluateObject(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (Object) (arguments[1].evaluateObject(i));
        else
            return (Object) (arguments[2].evaluateObject(i));
    }

    /**
    * Used to conditionally evaluate arguments of BYTE Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A byte representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final byte evaluateByte(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (byte) (arguments[1].evaluateByte(i));
        else
            return (byte) (arguments[2].evaluateByte(i));
    }

    /**
    * Used to conditionally evaluate arguments of SHORT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A short representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final short evaluateShort(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (short) (arguments[1].evaluateShort(i));
        else
            return (short) (arguments[2].evaluateShort(i));
    }

    /**
    * Used to conditionally evaluate arguments of INT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return An int representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final int evaluateInt(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (int) (arguments[1].evaluateInt(i));
        else
            return (int) (arguments[2].evaluateInt(i));
    }

    /**
    * Used to conditionally evaluate arguments of LONG Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A long representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final long evaluateLong(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (long) (arguments[1].evaluateLong(i));
        else
            return (long) (arguments[2].evaluateLong(i));
    }

    /**
    * Used to conditionally evaluate arguments of FLOAT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A float representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final float evaluateFloat(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (float) (arguments[1].evaluateFloat(i));
        else
            return (float) (arguments[2].evaluateFloat(i));
    }

    /**
    * Used to conditionally evaluate arguments of DOUBLE Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A double representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final double evaluateDouble(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (double) (arguments[1].evaluateDouble(i));
        else
            return (double) (arguments[2].evaluateDouble(i));
    }

    /**
    * Used to conditionally evaluate arguments of CHAR Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A char representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final char evaluateChar(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (char) (arguments[1].evaluateChar(i));
        else
            return (char) (arguments[2].evaluateChar(i));
    }

    /**
    * Used to conditionally evaluate arguments of BOOLEAN Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be
    *                   conditionally evaluated.
    * @return A boolean representing the result of the evaluation.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final boolean evaluateBoolean(Individual i, Gene[] arguments)
    throws EvaluationException {
        if (arguments[0].evaluateBoolean(i))
            return (boolean) (arguments[1].evaluateBoolean(i));
        else
            return (boolean) (arguments[2].evaluateBoolean(i));
    }

    /**
     * Creates an If Object of the same type.
     *
     * @return  A reference to the same Object since no instance varaibles need
     *          to be changed.
     */
    public final Primitive instance() {
        return this;
    }

    /**
    * Returns a String representation of this Function.
    *
    * @return A String containing the name of the Function.
    */
    public String toString() {
        return "If";
    }
}
