/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:24:33 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.primitives;

import gpsys.*;


/**
 * <pre>
 *  BOOLEAN EQ(&lt;typeX&gt; a, &lt;typeX&gt; b)
 * </pre>
 *
 * EQ is a generic function that tests the equality of its two arguments.
 * EQ needs to be Type instantiated during construction to work with particular
 * argument Types.  The supported argument Types include BYTE, SHORT, INT, LONG,
 * FLOAT, DOUBLE, CHAR and BOOLEAN.
 *
 * @see         gpsys.Type
 * @see         gpsys.Primitive
 * @see         gpsys.Function
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:24:33 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class EQ extends Function {

    /**
    * Constructs an EQ Function that works with the specified Type.  The
    * supported Types include BYTE, SHORT, INT, LONG, FLOAT, DOUBLE, CHAR and
    * BOOLEAN.
    *
    * @param type   The Type of this EQ Function.
    */
    public EQ(Type argTypes) {
        this.type = Type.BOOLEAN;
        this.argTypes = new Type[2];
        this.argTypes[0] = argTypes;
        this.argTypes[1] = argTypes;
    }

    /**
    * Used to test arguments of any of the supported Types for equality.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be tested.
    * @return   A boolean whose value is true if they are equal and false
    *           otherwise.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final boolean evaluateBoolean(Individual i, Gene[] arguments)
    throws EvaluationException {
        switch(argTypes[0].typeID) {
            case Type.BYTE_typeID:
                return
                    arguments[0].evaluateByte(i) ==
                    arguments[1].evaluateByte(i);
            case Type.SHORT_typeID:
                return
                    arguments[0].evaluateShort(i) ==
                    arguments[1].evaluateShort(i);
            case Type.INT_typeID:
                return
                    arguments[0].evaluateInt(i) ==
                    arguments[1].evaluateInt(i);
            case Type.LONG_typeID:
                return
                    arguments[0].evaluateLong(i) ==
                    arguments[1].evaluateLong(i);
            case Type.FLOAT_typeID:
                return
                    arguments[0].evaluateFloat(i) ==
                    arguments[1].evaluateFloat(i);
            case Type.DOUBLE_typeID:
                return
                    arguments[0].evaluateDouble(i) ==
                    arguments[1].evaluateDouble(i);
            case Type.CHAR_typeID:
                return
                    arguments[0].evaluateChar(i) ==
                    arguments[1].evaluateChar(i);
            case Type.BOOLEAN_typeID:
                return
                    arguments[0].evaluateBoolean(i) ==
                    arguments[1].evaluateBoolean(i);
            default:
                return
                    arguments[0].evaluateObject(i).equals(
                        arguments[1].evaluateObject(i)
                    );
        }
    }

    /**
    * Creates an instance of this EQ object.  The Type instantiation is
    * preserved so that if an INT EQ is being cloned, the clone will also be
    * INT instantiated.
    *
    * @return   A reference to this Object (yes the same object, since
    *           the type information is to be the same, there are therefore
    *           no instance variables to be modified.
    */
    public final Primitive instance() {
        return this;
    }

    /**
    * Returns a String representation of this Function.
    *
    * @return A String containing the name of the Function.
    */
    public String toString() {
        return "EQ";
    }
}
