/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:24:33 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.primitives;

import gpsys.*;


/**
 * <pre>
 *  &lt;typeX&gt; Div(&lt;typeX&gt; num1, &lt;typeX&gt; num2)
 * </pre>
 *
 * Div is a generic function that returns the result of dividing the first
 * argument by the second argument.  Div needs to be Type instantiated during
 * construction to work with particular argument Types.  The supported argument
 * Types include BYTE, SHORT, INT, LONG, FLOAT and DOUBLE.  Div generates a
 * DivisionByZero exception when the numerator is 0.
 *
 * @see         gpsys.Type
 * @see         gpsys.Primitive
 * @see         gpsys.Function
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:24:33 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Div extends Function {

    /**
    * Constructs a Div Function that works with the specified Type.  The
    * supported Types include BYTE, SHORT, INT, LONG, FLOAT and DOUBLE.
    *
    * @param type   The Type of this Div Function.
    */
    public Div(Type type) {
        this.type = type;
        argTypes = new Type[2];
        argTypes[0] = type;
        argTypes[1] = type;
    }

    /**
    * Used to divide arguments of BYTE Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be divided.
    * @return A byte representing the result of the division.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final byte evaluateByte(Individual i, Gene[] arguments)
    throws EvaluationException {
        byte divisor = arguments[1].evaluateByte(i);
        if (divisor == 0)
            throw new DivideByZeroException("Div:evaluateByte");
        return (byte) (arguments[0].evaluateByte(i) / divisor);
    }

    /**
    * Used to divide arguments of SHORT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be divided.
    * @return A short representing the result of the division.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final short evaluateShort(Individual i, Gene[] arguments)
    throws EvaluationException {
        short divisor = arguments[1].evaluateShort(i);
        if (divisor == 0)
            throw new DivideByZeroException("Div:evaluateShort");
        return (short) (arguments[0].evaluateShort(i) / divisor);
    }

    /**
    * Used to divide arguments of INT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be divided.
    * @return A int representing the result of the division.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final int evaluateInt(Individual i, Gene[] arguments)
    throws EvaluationException {
        int divisor = arguments[1].evaluateInt(i);
        if (divisor == 0)
            throw new DivideByZeroException("Div:evaluateInt");
        return arguments[0].evaluateInt(i) / divisor;
    }

    /**
    * Used to divide arguments of LONG Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be divided.
    * @return A long representing the result of the division.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final long evaluateLong(Individual i, Gene[] arguments)
    throws EvaluationException {
        long divisor = arguments[1].evaluateLong(i);
        if (divisor == 0)
            throw new DivideByZeroException("Div:evaluateLong");
        return arguments[0].evaluateLong(i) / divisor;
    }

    /**
    * Used to divide arguments of FLOAT Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be divided.
    * @return A float representing the result of the division.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final float evaluateFloat(Individual i, Gene[] arguments)
    throws EvaluationException {
        float divisor = arguments[1].evaluateFloat(i);
        if (divisor == 0.0f)
            throw new DivideByZeroException("Div:evaluateFloat");
        return arguments[0].evaluateFloat(i) / divisor;
    }

    /**
    * Used to divide arguments of DOUBLE Types.
    *
    * @param i          The individual being evaluated.
    * @param arguments  The Gene trees representing the arguments to be divided.
    * @return A double representing the result of the division.
    * @exception    EvaluationException If there is an evaluation failure.
    */
    public final double evaluateDouble(Individual i, Gene[] arguments)
    throws EvaluationException {
        double divisor = arguments[1].evaluateDouble(i);
        if (divisor == 0.0)
            throw new DivideByZeroException("Div:evaluateDouble");
        return arguments[0].evaluateDouble(i) / divisor;
    }

    /**
    * Creates an instance of this Div object.  The Type instantiation is
    * preserved so that if an INT Div is being cloned, the clone will also be
    * INT instantiated.
    *
    * @return   A reference to this Object (yes the same object, since
    *           the type information is to be the same, there are therefore
    *           no instance variables to be modified.
    */
    public final Primitive instance() {
        return this;
    }

    /**
    * Returns a String representation of this Function.
    *
    * @return A String containing the name of the Function.
    */
    public String toString() {
        return "Div";
    }
}
