/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:26:49 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.lawnmower;

/**
 * The lawnmower itself is a robot that moves around on a lawn, cutting it.
 * Unfortunately it only has a limited number of energy units available.
 * These are divided into a number of left turns and number of movements.  The
 * lawnmower is only able to turn left when there are left turn units
 * available, and similarly it can only move when there are movement units
 * available.  The lawnmower therefore stops when both of these units have run
 * out.  The lawn is mowed either by moving forward by one square thereby
 * mowing that square or by leap frogging to any square and also mowing it.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:26:49 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Mower implements java.io.Serializable {
    /**
     * The lawn to be mowed.
     */
    Lawn lawn;
    /**
     * The X  corrdinate of the mower on the lawn.
     */
    int x;
    /**
     * The Y  corrdinate of the mower on the lawn.
     */
    int y;
    /**
     * The current direction of the mower, can be one of NORTH, EAST, SOUTH or
     * WEST.
     */
    int direction;
    /**
     * The ration of left turn operations available.
     */
    int leftTurnsAvailable;
    /**
     * The ration of movement operations available.
     */
    int movementAvailable;

    /**
     * The value of direction when the mower is pointing NORTH.
     */
    public final static int NORTH   = 0;
    /**
     * The value of direction when the mower is pointing EAST.
     */
    public final static int EAST    = 1;
    /**
     * The value of direction when the mower is pointing SOUTH.
     */
    public final static int SOUTH   = 2;
    /**
     * The value of direction when the mower is pointing WEST.
     */
    public final static int WEST    = 3;

    /**
     * Construct a new mower to lawn the specified lawn.  The mower starts at
     * the default position (4,4) and direction NORTH.  The number of left
     * turn operations and the number of movement operations available are
     * both initialised to 100.
     *
     * @param lawn  The lawn to be mowed.
     */
    public Mower(Lawn lawn) {
        this.lawn = lawn;
        reset();
    }

    /**
     * Resets the mower state to default values (position = (4,4),
     * direction = NORTH, leftTurnsAvailable = 100, movementAvailable = 100.
     */
    public void reset() {
        x = 4;
        y = 4;
        direction = Mower.NORTH;
        leftTurnsAvailable = 100;
        movementAvailable  = 100;
    }

    /**
     * Turns the mower left if possible (if there are suffucient left turn
     * units).
     */
    public final void left() {
        if (leftTurnsAvailable == 0)
            return;
        switch (direction) {
            case Mower.NORTH:
                direction = Mower.WEST;
                break;
            case Mower.EAST:
                direction = Mower.NORTH;
                break;
            case Mower.SOUTH:
                direction = Mower.EAST;
                break;
            case Mower.WEST:
                direction = Mower.SOUTH;
                break;
        }
        leftTurnsAvailable--;
    }

    /**
     * Movement energy permitting, moves the mower forward by one square in the
     * current direction cutting the grass on that square.
     */
    public final void mow() {
        if (movementAvailable == 0)
            return;
        switch(direction) {
            case Mower.NORTH:
                y = (y == 0) ?  7 : (y - 1);
                break;
            case Mower.EAST:
                x = (x + 1) % 8;
                break;
            case Mower.SOUTH:
                y = (y + 1) % 8;
                break;
            case Mower.WEST:
                x = (x == 0) ?  7 : (x - 1);
                break;
        }
        lawn.cut(x,y);
        movementAvailable--;
    }

    /**
     * Movement energy permitting, leap frogs the mower to by the specified
     * (x,y) displacements.  The grass on square that the mower ends up on is
     * cut.
     *
     * @param displacement  The x and y displacements.
     * @return The displacement specified.
     */
    public final Vector2Mod8 frog(Vector2Mod8 displacement) {
        if (movementAvailable == 0)
            return displacement;
        x = (x + displacement.x) % 8;
        x = (x < 0) ? (x + 8) : x;
        y = (y + displacement.y) % 8;
        y = (y < 0) ? (y + 8) : y;
        lawn.cut(x,y);
        movementAvailable--;
        return displacement;
    }
}
