/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:26:49 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.lawnmower;

import gpsys.*;
import gpsys.primitives.*;

/**
 * The Fitness class for the lawnmower problem.  The fitness is
 * measured using two values.  The first is the number of squares on the lawn
 * left uncut when the evolved program has complted.   The second value is
 * the complexity of the GP program.  The fitness of one GP is is deemed better
 * than another if either the error is smaller or if the error is equal, but
 * the complexity is smaller.  The termination criteria is met when the number
 * sqaures left uncut is zero.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:26:49 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class LawnMowerFitness extends Fitness {
    /**
     * The number of uncut squares left on the lawn.
     */
    int     fitness;
    /**
     * The complexity of the individual i.e. the number of Genes it contains.
     */
    int     complexity;
    /**
     * The lawn to be cut.
     */
    Lawn    lawn;
    /*
     * The mower to be controlled by the program.
     */
    Mower   mower;

    /**
     * Construct a Fitness object with default Fitness.
     */
    public LawnMowerFitness(Lawn lawn, Mower mower) {
        this.lawn  = lawn;
        this.mower = mower;
        fitness = 0;
        complexity  = 0;
    }

    /**
     * Constructs a Fitness object by evaluating the specified individual.
     * The raw fitness is the number of squares on the lawn left uncut after
     * the GP evolved code has executed.
     *
     * @param lawn  The lawn to be cut by the mower.
     * @param mower The lawnmower to use to cut the grass.
     * @param gpParameters The GP parameters for this run.
     * @param i The individual to be evaluated.
     */
    public LawnMowerFitness(Lawn lawn, Mower mower,
        GPParameters gpParameters, Individual i) {

        this.lawn  = lawn;
        this.mower = mower;

        lawn.grow();
        mower.reset();

        try {
            i.evaluateObject();
        }
        catch (GPException e) {
            gpParameters.observer.exception(e);
        }

        fitness = 64 - lawn.cutCount;
        complexity = i.complexity();
    }

    /**
     * Adds a fitness value to this fitness value.
     *
     * @param f the fitness value to be added.
     */
    public void add(Fitness f) {
        fitness += ((LawnMowerFitness)f).fitness;
        complexity  += ((LawnMowerFitness)f).complexity;
    };

    /**
     * Divide the fitness by the specified integer.  This is used by the
     * GP system to calculate the average fitness of the population.
     *
     * @param   divisor the integer to divide the fitness by.
     */
    public void divide(int divisor) {
        fitness /= divisor;
        complexity  /= divisor;
    }

    /**
     * Tests if this fitness value is greater than another fitness value.
     *
     * @param   f   the fitness with which to compare.
     * @return  true if f has higher fitness, false otherwise.
     */
    public boolean greaterThan(Fitness f) {
        if (fitness < ((LawnMowerFitness)f).fitness)
            return true;
        if (fitness == ((LawnMowerFitness)f).fitness)
            return complexity  < ((LawnMowerFitness)f).complexity;
        return false;
    }

    /**
     * Tests if this fitness is less than another fitness.
     *
     * @param   f   the fitness with which to compare.
     * @return  true if f has less fitness, false otherwise.
     */
    public boolean lessThan(Fitness f) {
        if (fitness > ((LawnMowerFitness)f).fitness)
            return true;
        if (fitness == ((LawnMowerFitness)f).fitness)
            return complexity  > ((LawnMowerFitness)f).complexity;
        return false;
    }

    /**
     * Tests if this fitness is equal to another fitness.
     *
     * @param   f   the fitness with which to compare.
     * @return  true if f has the same fitness, false otherwise.
     */
    public boolean equals(Fitness f) {
        return
            (fitness == ((LawnMowerFitness)f).fitness) &&
            (complexity  == ((LawnMowerFitness)f).complexity);
    }

    /**
     * Creates a new instance of the Fitness object with a default fitness.
     * Note that the same Lawn and Mower are used in the new instance.
     *
     * @return  an instance of a Fitness object with default fitness.
     */
    public Fitness instance() {
        return new LawnMowerFitness(lawn, mower);
    }

    /**
     * Creates a new instance of the Fitness object which represents the
     * fitness of the specified individual.
     * Note that the same Lawn and Mower are used in the new instance.
     *
     * @param   gpParameters    the parameters for this GP run.
     * @param   i               the individual to be evaluated.
     * @return  The Fitness of the specified individual.
     */
    public Fitness instance(GPParameters gpParameters, Individual i) {
        return new LawnMowerFitness(lawn, mower, gpParameters, i);
    }

    /**
     * Tests whether this fitness meets the termination criteria.
     *
     * @return  true if the termination criteria has been met, false otherwise.
     */
    public boolean terminationCondition() {
        return fitness == 0;
    }

    /**
     * Converts the fitness into a String suitable for printing.
     *
     * @return  A String representing the fitness.
     */
    public String toString() {
        return "Fitness(" + fitness + "," + complexity + ")";
    }
}
