/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:26:49 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.lawnmower;

import gpsys.*;
import java.io.*;
import java.awt.*;

/**
 * The user interface for the lawn mower problem.  This class
 * also contains the main() function to execute the application.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:26:49 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class LawnMower extends Frame implements GPObserver {
    /**
     * The lawn being mowed.
     */
    Lawn lawn;
    /**
     * The file prefix to use for saving generations and writing reports.
     */
    String filePrefix;

    /**
     * Construct a lawn mower user interface.
     *
     * @param lawn  The lawn being mowed.
     * @param lawn  The file prefix to use for saving generations and writing
     *              reports.
     */
    public LawnMower(Lawn lawn, String filePrefix) {
        super("Lawn Mower");                // sets the title of the frame

        this.filePrefix = filePrefix;
        this.lawn = lawn;

        // a lawn viewer allows us to visualise the state of the lawn.
        LawnViewer lawnViewer = new LawnViewer(lawn);
        lawn.setObserver(lawnViewer);
        add("Center", lawnViewer);
        setSize(400,420);
    }

    /**
     * If the filePrefix is null, just write a report of the current generation
     * to the standard output.  Otherwise, also append the report to the file
     * "filePrefix.txt" and save the current generation to the file
     * "filePrefix.p1.gzip".  If the termination criteria has been met, or the
     * maximum number of generations have been reached, visualise the
     * best lawn mower in action!!!
     *
     * @param gpParameters  The GP parameters used for this run.
     * @param how           How the generation was created.  Can be either
     *                      CREATION, FROMSTREAM or EVOLVED.
     */
    public void generationUpdate(GPParameters gpParameters, int how) {

        // if a file prefix was given, write the report to file and save the
        // current generation.  This need not be done if the generation was
        // just loaded from file.
        if ((filePrefix != null) &&
            (how == GPObserver.CREATION || how == GPObserver.EVOLVED)) {
            try {
                diagnosticUpdate("Saving current generation...");
                gpParameters.save(filePrefix);
                diagnosticUpdate("Saved  current generation.");
            }
            catch (IOException e) {
                System.out.println("gpParameters.save() : " + e);
                System.exit(1);
            }

            try {
                gpParameters.writeReport(filePrefix,how == GPObserver.CREATION);
            }
            catch (IOException e) {
                System.out.println("gpParameters.writeReport() : " + e);
                System.exit(1);
            }
        }

        // now write a report of the current population to the standard output
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        gpParameters.writeReport(pw,
            (how == GPObserver.CREATION) || (how == GPObserver.FROMSTREAM));
        System.out.print(sw.toString());

        // if the termination criteria has been reached or the maximum number of
        // generations have been evolved, visualise the best mower in action.
        if (gpParameters.population.bestRun.fitness.terminationCondition() ||
            (gpParameters.population.generation ==
                (gpParameters.generations - 1))) {
            setVisible(true);
            lawn.enableObserver();
            gpParameters.fitness.instance(
                gpParameters, gpParameters.population.bestRun);
            lawn.disableObserver();
            System.out.println("GP run completed..");
            System.exit(0);
        }
    }

    /**
     * We are not interested in this update, so we just ignore it.
     *
     * @param   gpParameters    the GP parameters used for this run.
     * @param   i               the Individual that has just been created.
     * @param   creationIndex   the index of the Individual in the population.
     * 
     */
    public void individualUpdate(GPParameters gpParameters,
        Individual i, int creationIndex) {
    }

    /**
     * We are not interested in this update, so we just ignore it.
     *
     * @param   gpParameters    the GP parameters used for this run.
     * @param   individualIndex the index of the created Individual in the
     *                          population.
     * @param   creationMethod  how the Individual was created.  Can be either
     *                          VIA_MUTATION or VIA_CROSSOVER.
     */
    public void individualUpdate(GPParameters gpParameters,
        int individualIndex, int creationMethod) {
    }

    /**
     * Print the diagnostic message to the standard output.
     *
     * @param s The diagnostic message.
     */
    public void diagnosticUpdate(String s) {
        System.out.println(s);
    }

    /**
     * Print the exception and generate a stack trace on the standard output.
     *
     * @param e The exception that was genearted.
     */
    public void exception(GPException e) {
        System.out.println(e.getMessage());
        e.printStackTrace();
        System.exit(0);
    }

    /**
     * The main() method of the lawn mower application.  This
     * application may be invoked in any of the following ways.
     * <pre>
     *      java LawnMower &lt;filePrefix&gt;
     *      java LawnMower &lt;filePrefix&gt; &lt;generations&gt;
     *      java LawnMower &lt;filePrefix&gt; &lt;RNG seed&gt; &lt;population&gt; &lt;generations&gt;
     *      java LawnMower &lt;RNG seed&gt; &lt;population&gt; &lt;generations&gt;
     * </pre>
     * 
     * The first istructs the application to restart from the last saved
     * session using the files with the prefix specified.  The second is the
     * same as the first, execept that the maximum number of geneartion is
     * modified as specified.  The third istructs the application to start
     * a new run using the specifed file prefix for saves, and the
     * specified parameters for the run.  The last is the similar to the
     * previous invokation except that nothing is saved to disk, and is hence
     * very fast.
     */
    public static void main(String[] argv) {

        // change this to whatever is suitable to you!!
        java.util.Locale.setDefault(java.util.Locale.UK);

        LawnMowerGPParameters gpParameters = null;
        String filePrefix = null;

        // now read the command line arguments
        switch(argv.length) {
            case 1: // load entirely from saved session
            case 2: // load from saved session but change the max generations
                filePrefix = argv[0];
                try {
                    System.out.println("Loading last saved generation...");
                    gpParameters =
                        (LawnMowerGPParameters) GPParameters.load(filePrefix);
                    if (gpParameters != null)
                        System.out.println("Loaded  last saved generation...");
                }
                catch (java.io.IOException e) {
                    System.out.println("Loading problem : " + e);
                    System.exit(1);
                }
                catch (ClassNotFoundException e) {
                    System.out.println("Loading problem : " + e);
                    System.exit(1);
                }
                if (argv.length == 2) {
                    int generations = Integer.parseInt(argv[1]);
                    gpParameters.generations = generations;
                }
                break;
            case 3: {
                    long rngSeed    = Long.parseLong(argv[0]);
                    int population  = Integer.parseInt(argv[1]);
                    int generations = Integer.parseInt(argv[2]);
                    gpParameters = new
                        LawnMowerGPParameters(rngSeed, population, generations);
                }
                break;
            case 4: {
                    filePrefix = argv[0];
                    long rngSeed    = Long.parseLong(argv[1]);
                    int population  = Integer.parseInt(argv[2]);
                    int generations = Integer.parseInt(argv[3]);
                    gpParameters = new
                        LawnMowerGPParameters(rngSeed, population, generations);
                }
                break;
            default:
                System.out.println("Usage : LawnMower <file>");
                System.out.println("        LawnMower <file> <generations>");
                System.out.println("        LawnMower " +
                    "<RNG seed> <population> <generations>");
                System.out.println("        LawnMower " +
                    "<file> <RNG seed> <population> <generations>");
                return;
        }

        // set the observer to be an instance of our user interface
        gpParameters.observer = new
            LawnMower(gpParameters.lawn, filePrefix);

        // create a new GP system
        GPsys gpSys = new GPsys(gpParameters);
        // and start evolving !!!
        gpSys.evolve();
    }
}
