/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:26:49 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys.lawnmower;

/**
 * The lawn consists of a torroidal grid of squares upon which the "grass
 * grows".  The lawn can be grown, and cut.  In addition, an observer can be
 * set which gets update messages each time the state of the lawn changes.
 *
 * @see gpsys.lawnmower.LawnObserver
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:26:49 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Lawn implements java.io.Serializable {
    /**
     * Used to hold the state of each square of the lawn.
     */
    int[][] lawn;
    /**
     * Used to keep a count of how many squares have been cut.
     */
    int cutCount;
    /**
     * True if the observer is wants to received state change information.
     */
    transient boolean       observed;
    /**
     * A reference to an observer interested in monitoring the state of this
     * lawn.
     */
    transient LawnObserver  observer;

    /**
     * Used to tag squares on the lawn as having been cut.
     */
    public static final int CUT     = 0;
    /**
     * Used to tag squares as not having been cut.
     */
    public static final int UNCUT   = 1;

    /**
     * Create a square lawn of the specified dimension, with no observer.
     *
     * @param dimension The value to be used for both the height and width
     *                  of the lawn.
     */
    public Lawn(int dimension) {
        lawn = new int[dimension][dimension];
        observer = null;
        observed = false;
    }

    /**
     * Set the observer of the lawn to the specified observer.
     *
     * @param lo    The observer of the lawn.
     */
    public void setObserver(LawnObserver lo) {
        observer = lo;
    }

    /**
     * Sets the all of the squares on the lawn to UNCUT state (grown).  The
     * observer is notified of this change.
     */
    public final void grow() {
        for (int i = 0; i < lawn.length; i++)
            for (int j = 0; j < lawn.length; j++)
                lawn[i][j] = UNCUT;
        cutCount = 0;
        if (observed)
            observer.lawnGrown();
    }

    /**
     * Sets the specified square on the lawn to the CUT state, and informs the
     * observer of the event.
     *
     * @param x The X coordinate of the square to be cut.
     * @param y The Y coordinate of the square to be cut.
     */
    public final void cut(int x, int y) {
        if (lawn[x][y] == UNCUT) {
            lawn[x][y] = CUT;
            cutCount++;
            if (observed)
                observer.lawnCutting(x, y);
        }
    }

    /**
     * Gets a count of the number of squares on the lawn in the CUT state.
     *
     * @return The number of squares on the lawn in the CUT state.
     */
    public final int cutCount() {
        return cutCount;
    }

    /**
     * Enables the observer to receive change messages.
     */
    public void enableObserver() {
        observed = true;
    }

    /**
     * Disables the observer from receiving change messages.
     */
    public void disableObserver() {
        observed = false;
    }
}
