/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

import java.util.*;

/**
 * A Population holds all the Individuals of a generation.  Additional
 * information is maintained such as the current generation number, the
 * average fitness and complexity of the Individuals, and the best individual
 * in the Population.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Population implements java.io.Serializable {
    /**
     * The Individuals in this population.
     */
    public Individual[]     p;

    /**
     * The current generation number.
     */
    public int              generation;

    /**
     * The fittest Individual in the Population.
     */
    public Individual       bestGeneration;

    /**
     * The fittest Individual of the Run.
     */
    public Individual       bestRun;

    /**
     * The average fitness of the Population.
     */
    public Fitness          averageFitness;

    /**
     * The average number of nodes in the Individuals of the Population.
     */
    public double           averageComplexity;

    /**
     * The GP parameters for the problem.
     */
    public GPParameters     gpParameters;

    /**
     * Total number of reproductions performed.
     */
    public long reproductions;

    /**
     * Total number of mutations performed.
     */
    public long mutations;

    /**
     * Total number of crossovers performed.
     */
    public long crossovers;

    /**
     * Used for implementing a minimal memory generational evolution engine.
     */
    transient private CrossoverBookkeeping bookkeepingInfo;

    /**
     * Used internally to signal a reproduction operation.
     */
    private static final int REPRODUCTION   = 0;

    /**
     * Used internally to signal a mutation operation.
     */
    private static final int MUTATION       = 1;

    /**
     * Used internally to signal a crossover operation.
     */
    private static final int CROSSOVER      = 2;

    /*
     * Create a population of Individuals for the given the problem.
     *
     * @param   gpParameters    The parameters for the problem.
     */
    public Population(GPParameters gpParameters) {
        this.gpParameters = gpParameters;
        generation = 0;
        // create each Individual, telling the observer after each is created
        p = new Individual[gpParameters.populationSize];
        for (gpParameters.creationIndex = 0;
             gpParameters.creationIndex < gpParameters.populationSize;
             gpParameters.creationIndex++) {
            p[gpParameters.creationIndex] = new Individual(gpParameters);
            gpParameters.observer.individualUpdate(gpParameters,
                p[gpParameters.creationIndex], gpParameters.creationIndex);
        }
        // now update the statistics for population
        updateStats();
    }

    /**
     * Evolve the current population via the steady state or generational
     * engine depending on the value of the engine parameter.
     */
    public void evolve() {
        switch (gpParameters.engine) {
            case gpParameters.ENGINE_STEADYSTATE:
                evolveSteadyState();
                break;
            case gpParameters.ENGINE_GENERATIONAL:
                evolveGenerational();
                break;
            default:
                break;
        }
    }


    /*
     * Evolve the Population by one generation using a steady state engine.
     * The steady state engine makes each new individual immediately available
     * to the population, so that a new so that a generation is actually the
     * point at which an entire population has been created.
     */
    public void evolveSteadyState() {
        for (int i = 0; i < p.length; i++) {
            int mother;
            int father;
            int child;
            // depending on the biases, select one of the genetic operators
            int opSelector = selectGeneticOperation();
            if (opSelector == REPRODUCTION) {
                mother = selectBest();
                child = selectWorst();
                p[child] = executeReproductionOperation(mother);
            }
            else if (opSelector == MUTATION) {
                // get indices to mum + the one to be replaced by child
                mother = selectBest();
                child = selectWorst();
                p[child] = executeMutationOperation(mother);
            }
            else {// if (opselector == CROSSOVER)
                // get indices to mum & dad + the one to be replaced by child
                mother = selectBest();
                father = selectBest();
                child = selectWorst();
                // mum & dad do crossover to create a child
                p[child] = executeCrossoverOperation(mother, father);
            }
        }
        updateStats();
        generation++;
    }


    /*
     * Evolve the Population by one generation using a generational engine.
     * Members of the new population are not available until the entire
     * population has been generated.
     */
    public void evolveGenerational() {
        // initialise the bookkeeping data structure
        if (bookkeepingInfo == null)
            bookkeepingInfo = new CrossoverBookkeeping(p.length);
        else
            bookkeepingInfo.reset();

        // choose the operations that will create the new population

        int reproductions = 0;
        int mutations = 0;
        int crossovers = 0;
        int mother;
        int father;
        // for each individual to be created
        for (int i = 0; i < p.length; i++) {
            int[] selected;
            // depending on the biases, select one of the genetic operators
            int opSelector = selectGeneticOperation();
            // make a note of which operation was selected + the parents
            if (opSelector == REPRODUCTION) {
                // reproduce
                reproductions++;
                mother = selectBest();
                bookkeepingInfo.addReproductionOperation(mother);
            }
            else if (opSelector == MUTATION) {
                // mutate
                mutations++;
                mother = selectBest();
                bookkeepingInfo.addMutationOperation(mother);
            }
            else {// if (opselector == CROSSOVER)
                // cross
                crossovers++;
                mother = selectBest();
                father = selectBest();
                bookkeepingInfo.addCrossoverOperation(mother, father);
            }
        }
        
        bookkeepingInfo.createLists();

        // now create the new population using the chosen operations

        Individual temp1 = null;
        Individual temp2 = null;

        // loop through and execute each operation previously chosen
        for (int i = 0; i < p.length; i++) {
            GeneticOperation op = bookkeepingInfo.removeOperation();
            Individual result;
            if (op instanceof ReproductionOperation)
                result = executeReproductionOperation(op.parent1);
            else if (op instanceof MutationOperation)
                result = executeMutationOperation(op.parent1);
            else { // if (op instanceof CrossoverOperation)
                CrossoverOperation cx = (CrossoverOperation) op;
                result = executeCrossoverOperation(cx.parent1, cx.parent2);
            }

            int dead = bookkeepingInfo.removeFreeSlot();

            // if the free slot chosen is not a member of the population
            // use one of the two free pegs else replace the dead individual
            // with the new one.
            if (dead >= p.length) {
                if (dead == p.length)
                    temp1 = result;
                else
                    temp2 = result;
            } else
                p[dead] = result;
        }

        // now move the individual from the extra slots into dead individuals
        if (temp1 != null || temp2 != null) {
            int dead;
            if (temp1 != null) {
                dead = bookkeepingInfo.removeFreeSlot();
                p[dead] = temp1;
            }
            if (temp2 != null) {
                dead = bookkeepingInfo.removeFreeSlot();
                p[dead] = temp2;
            }
        }

        // this is convenient point to garbage collect !!
        System.gc();

        updateStats();
        generation++;
    }

    /**
     * Execute a crossover operation using specified parents.
     *
     * @param   mum     index of the first parent involved in the crossover.
     * @param   father  index of the secong parent involved in the crossover.
     *
     * @return  the child resulting from the crossover.
     */
    private Individual executeCrossoverOperation(int mum, int dad) {
        Individual newIndividual =  new
            Individual(gpParameters, p[mum], p[dad]);
        gpParameters.observer.individualUpdate(gpParameters,
            newIndividual, GPObserver.VIA_CROSSOVER);
        return newIndividual;
    }

    /**
     * Execute a mutation operation using specified parent.
     *
     * @param   mum     index of the parent involved in the mutation.
     *
     * @return  the child resulting from a mutation of the parent.
     */
    private Individual executeMutationOperation(int mum) {
        Individual newIndividual = new Individual(gpParameters, p[mum], true);
        gpParameters.observer.individualUpdate(gpParameters,
            newIndividual, GPObserver.VIA_MUTATION);
        return newIndividual;
    }

    /**
     * Execute a reproduction operation using the specified parent.
     *
     * @param   mum     index of the parent involved in the mutation.
     *
     * @return  the child that is a reproduction of the parent.
     */
    private Individual executeReproductionOperation(int mum) {
        Individual newIndividual = new Individual(gpParameters, p[mum], false);
        gpParameters.observer.individualUpdate(gpParameters,
            newIndividual, GPObserver.VIA_REPRODUCTION);
        return newIndividual;
    }

    /**
     * Randomly select a new Genetic Operation to be performed using the
     * selection probabilities specified in the GPParameters object via
     * the variable pMutation and pReproduction.  The probability of selecting
     * a crossover operation is 1 - (pMutation + pReproduction).
     *
     * @return  Either REPRODUCTION, MUTATION or CROSSOVER.
     */
    private int selectGeneticOperation() {
        double opSelector = gpParameters.rng.nextDouble();
        if (opSelector < gpParameters.pReproduction)
            return REPRODUCTION;
        if (opSelector < (gpParameters.pReproduction + gpParameters.pMutation))
            return MUTATION;
        // if (opselector >= pReproduction + pMutation
        return CROSSOVER;
    }

    /**
     * Select the best individual from a tournament.
     *
     * @return  The index of the best individual from a tournament.
     */
    private int selectBest() {
        int[] tournament = tournamentSelect();
        return  tournament[0];
    }

    /**
     * Select the worst individual from a tournament.
     *
     * @return  The index of the worst individual from a tournament.
     */
    private int selectWorst() {
        int[] tournament = tournamentSelect();
        return tournament[gpParameters.tournamentSize - 1];
    }

    /**
     * Perform tournament selection.  N (specified by the tournament size
     * in the GP parameters) unique individuals are selected at random, and 
     * sorted in order of Fitness (hightest first).
     *
     * @return  An array of indices to N Individuals, sorted by fitness.
     */
    private int[] tournamentSelect() {
        int[] tournament = new int[gpParameters.tournamentSize];
        for (int i = 0; i < gpParameters.tournamentSize; i++) {
            // tournament size should be < population size
            boolean notUnique;
            do {
                notUnique = false;
                int tmp = gpParameters.rng.nextInt() % p.length;
                tournament[i] = (tmp < 0) ? -tmp : tmp;
                for (int j = 0; j < i; j++)
                    if (tournament[j] == tournament[i]) {
                        notUnique = true;
                        break;
                    }
            } while (notUnique);
        }
        sort(tournament);
        return tournament;
    }


    /**
     * Sort an array of indices to Individuals so that the first index
     * references the best Individual, and the last the worst.
     *
     * @param   The array of indices to Individuals to be sorted.
     */
    private void sort(int[] indexList) {
        // used for tournament selection (<= 7 items) hence very simple
        for (int i = 0; i < indexList.length; i++)
            for (int j = i; j < indexList.length; j++)
                if (p[ indexList[i] ].fitness.lessThan(
                        p[ indexList[j] ].fitness)) {
                    int tmp = indexList[i];
                    indexList[i] = indexList[j];
                    indexList[j] = tmp;
                }
    }

    /**
     * Update the statistics for the Population.  This involves finding the
     * individual with the highest fitness, calculating the average fitness
     * over the population and calculating the average complexity  over the
     * population.
     */
    public void updateStats() {
        // find the individual with the highest fitness
        // calculate the average fitness over the population
        // calculate the average complexity  over the population

        bestGeneration              = p[0];
        averageComplexity   = bestGeneration.complexity();
        averageFitness      = bestGeneration.fitness.instance();
        for (int i = 1; i < p.length; i++) {
            if (bestGeneration.fitness.lessThan(p[i].fitness))
                bestGeneration = p[i];
            averageFitness.add(p[i].fitness);
            averageComplexity += p[i].complexity();
        }
        averageFitness.divide(p.length);
        averageComplexity /= p.length;
        if (bestRun == null || bestRun.fitness.lessThan(bestGeneration.fitness))
            bestRun = bestGeneration;
    }


    /**
     * Print a report of the population using the specified PrintWriter.  The
     * report includes the generation number just completed, the date and time
     * of completion, the average Fitness, the average complexity and a dump
     * of the bestGeneration Individual.
     *
     * @param The PrintWriter to use to write the report.
     * 
     */
    public void report(java.io.PrintWriter pw) {
        pw.println("Generation " + generation + " completed...");
        pw.println("Date :- " +
            java.text.DateFormat.getDateTimeInstance().format(new Date()));
        pw.println("Average fitness " + averageFitness);
        pw.println("Average complexity " + averageComplexity);
        pw.println("best individual of generation = ");
        pw.println(bestGeneration);
        pw.println("best individual of run = ");
        pw.println(bestRun);
    }

}
