/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

import java.util.*;


/**
 * This class is used to hold information about the genetic operations
 * involving a a particular parent.  The class is used in the implementation
 * of memory efficient crossover as described in Genetic Programming III.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class ParentInfo {
    /**
     * A count of how many mutation operations involving this parent.
     */
    int mutationCount;
    
    /**
     * A count of how many reproduction operations involving this parent.
     */
    int reproductionCount;
    
    /**
     * A list containing information about parents involved in crossover
     * operations with this parent.
     */
    LinkedList crossoverParents;

    /**
     * A count of the total number of genetic operations involving this parent.
     */
    int totalOperationsCount;

    /**
     * The index of this parent in the population.
     */
    int id;

    /**
     * The state of this parent.  The state can be any one of the three
     * predefined states, which are DEAD, FREE and BUSY.
     */
    int state;

    /**
     * A parent has this state if it has expired and been replaced a new
     * Individual.
     */
    static final int DEAD   = 0;
    
    /**
     * A parent is deemed free if it has no outstanding genetic operations and
     * it has not already been replaced in the population.
     */
    static final int FREE   = 1;
    
    /**
     * A parent is busy if it has one or more outstanding genetic operations
     * to be performed.
     */
    static final int BUSY   = 2;

    /**
     * Points to the previous parent information block in the list that this
     * parent is contained within.
     */
    ParentInfo prev;
    
    /**
     * Points to the next parent information block in the list that this parent
     * is contained within.
     */
    ParentInfo next;

    /**
     * Constructs a ParentInfo data object for the specified parent.
     *
     * @param   id  The index of the individual for which parenting information
     *              is maintained.
     */
    public ParentInfo(int id) {
        this.id = id;
        state = FREE;
        crossoverParents = new LinkedList();
    }

    /**
     * Adds a crossover operation to this parent.
     *
     * @param   parent2 The second parent involved in the crossover.
     *
     * @return  A count of the total operations remaining for this parent.
     */
    public int addCrossover(ParentInfo parent2) {
        crossoverParents.add(parent2);
        state = BUSY;
        return ++totalOperationsCount;
    }

    /**
     * Adds a mutation operation to this parent.
     *
     * @return  A count of the total operations remaining for this parent.
     */
     public int addMutation() {
        mutationCount++;
        state = BUSY;
        return ++totalOperationsCount;
    }
    
    /**
     * Adds a reproduction operation to this parent.
     *
     * @return  A count of the total operations remaining for this parent.
     */
    public int addReproduction() {
        reproductionCount++;
        state = BUSY;
        return ++totalOperationsCount;
    }

    /**
     * Remove an operation from this parent.  Operations involving just one
     * parent are always removed first.
     *
     * @return  A genetic operation, which is currently either a mutation,
     *          reproduction or crossover operation.
     */
    public GeneticOperation removeOperation() {
        totalOperationsCount--;
        if (totalOperationsCount == 0)
            state = FREE;
        if (mutationCount != 0) {
            mutationCount--;
            return new MutationOperation(this.id);
        }
        else if (reproductionCount != 0) {
            reproductionCount--;
            return new ReproductionOperation(this.id);
        }
        else {
            ParentInfo parent2 = (ParentInfo) crossoverParents.removeFirst();
            return new CrossoverOperation(this.id, parent2.id);
        }
    }

    /**
     * Remove the crossover operation involving the specified second parent.
     *
     * @param parent2   The second parent involved in the crossover operation.
     */
    public void removeCrossover(ParentInfo parent2) {
        crossoverParents.remove(parent2);
        totalOperationsCount--;
        if (totalOperationsCount == 0)
            state = FREE;
    }

    /**
     * Get a count of the total number of genetic operations involving this
     * parent.
     */
    public int getOperations() {
        return totalOperationsCount;
    }

    /**
     * Get the index of this individual in the population.
     *
     * @return  The indentifier for this parent which equals it's index in the
     *          population.
     */
    public int getId() {
        return id;
    }

    /**
     * Kill this individual, which just sets the state to DEAD.
     */
    public void kill() {
        state = DEAD;
    }

    /**
     * Reset this instance.  This involves disassociating from it from any of
     * the four lists and clearing genetic operation information.
     */
    public void reset() {
        next = prev = null;
        mutationCount = reproductionCount = 0;
        crossoverParents.clear();
        totalOperationsCount = 0;
        state = FREE;
    }

    /**
     * Creates a string representing the state of this object.
     *
     * @return A String representing the state of this object.
     */
    public String toString() {
        String s = "";
        s += "[id=" + id + " mutationCount=" + mutationCount +
            " reproductionCount=" + reproductionCount +
            " totalOperationsCount=" + totalOperationsCount +
            " state=" + state + " ";
        ListIterator li = crossoverParents.listIterator();
        s += "xoverParents=[";
        while (li.hasNext()) {
            ParentInfo p = (ParentInfo) li.next();
            s += " " + p.id;
        }
        s += " ]]";
        return s;
    }
}
