/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

/**
 * An Individual is an evolved GP program.  It consists of a number of
 * Chromosomes (which are automatically defined functions).  An Individual has
 * fitness which measures the suitability of the Individual for solving the
 * problem for which it was evolved.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Individual implements java.io.Serializable {
    /**
     * An Individual has a number of Chromosomes.  The chromosome are Gene
     * trees representing automatically defined functions.
     */
    public  Chromosome[]    adf;

    /**
     * An Individual has Fitness which is a measure of the suitability of the
     * for solving the problem for which it was evolved.
     */
    public  Fitness         fitness;

    /**
     * The total number of Genes used across all Chromosomes.
     */
    public  int             complexity;

    /**
     * Create a completely new Individual.  This method creates Gene trees for
     * each Chromosome (ADF), and then evaluates the individual's fitness.
     *
     * @param gpParameters  the problem specific parameters.
     */
    public Individual(GPParameters gpParameters) {
        // allocate an array for the required number of ADFs
        adf = new Chromosome[gpParameters.adf.length];

        // create new ADFs and sum their complexity
        complexity = 0;
        for (int i = 0; i < gpParameters.adf.length; i++) {
            try {
                adf[i] = new Chromosome(gpParameters, i);
                complexity += adf[i].treeTop.complexity();
            }
            catch (TypeException e) {
                gpParameters.observer.exception(e);
            }
        }

        // now measure the fitness of the individual
        fitness = gpParameters.fitness.instance(gpParameters, this);
    }

    /**
     * Create a new Individual via reproduction or mutation of the mother.
     * For reproduction, the new individual is a clone of the mother.  For
     * mutation the new individual is a muation of the mother.  Mutation
     * involves replacing one random subtree in one Chromosome (ADF) with
     * a newly created type compatible one.
     *
     * @param gpParameters  the problem specific parameters.
     * @param mum           the mother.
     * @param mutate        whether to mutate or just reproduce
     */
    public Individual(GPParameters gpParameters, Individual mum,
        boolean mutate) {

        // allocate an array for the required number of ADFs
        adf = new Chromosome[gpParameters.adf.length];

        int mutantAdf;
        if (mutate)
            // pick the adf to be mutated
            mutantAdf = gpParameters.rng.nextInt(gpParameters.adf.length);
        else
            mutantAdf = -1;
        
        // create the new ADFs by cloning and/or mutating,
        // and then sum their complexity
        complexity = 0;
        for (int i = 0; i < gpParameters.adf.length; i++) {
            if (mutate && i == mutantAdf)
                adf[i] = Chromosome.mutate(mum.adf[i]);
            else
                adf[i] = mum.adf[i].deepClone();
            complexity += adf[i].treeTop.complexity();
        }
        // now measure the fitness of the individual
        // we need to do this even if don't mutate since the test cases
        // can change per generation
        fitness = gpParameters.fitness.instance(gpParameters, this);
    }


    /**
     * Create a new Individual via sexual reproduction (crossover).  The mother
     * is cloned and the clone is crossed with the father to create the child.
     *
     * @param gpParameters  the problem specific parameters.
     * @param mum           the mother.
     * @param dad           the father.
     */
    public Individual(GPParameters gpParameters,
        Individual mum, Individual dad) { 

        // allocate an array for the required number of ADFs
        adf = new Chromosome[gpParameters.adf.length];

        // decide which adf is going to take part in the crossover
        int crossoverAdf = gpParameters.rng.nextInt(gpParameters.adf.length);

        // create new Chromosomes via crossover of the mother and father
        // Chromosomes and sum their complexities.
        complexity = 0;
        for (int i = 0; i < gpParameters.adf.length; i++) {
            if (i == crossoverAdf)
                adf[i] = Chromosome.cross(mum.adf[i], dad.adf[i]);
            else
                adf[i] = mum.adf[i].deepClone();
            complexity += adf[i].complexity();
        }
        // now measure the fitness of the individual
        fitness = gpParameters.fitness.instance(gpParameters, this);
    }

    /**
     * Evaluate this Individual as a program returning an Object reference.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  An Object which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public Object evaluateObject() throws EvaluationException {
        return adf[0].evaluateObject(this);
    }

    /**
     * Evaluate this Individual as a program returning a byte.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  A byte which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public byte evaluateByte() throws EvaluationException {
        return adf[0].evaluateByte(this);
    }

    /**
     * Evaluate this Individual as a program returning a short.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  A short which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public short evaluateShort() throws EvaluationException {
        return adf[0].evaluateShort(this);
    }

    /**
     * Evaluate this Individual as a program returning an int.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  An int which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public int evaluateInt() throws EvaluationException {
        return adf[0].evaluateInt(this);
    }

    /**
     * Evaluate this Individual as a program returning a long.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     # @return  A long which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public long evaluateLong() throws EvaluationException {
        return adf[0].evaluateLong(this);
    }

    /**
     * Evaluate this Individual as a program returning a float.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  A float which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public float evaluateFloat() throws EvaluationException {
        return adf[0].evaluateFloat(this);
    }

    /**
     * Evaluate this Individual as a program returning a double.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  A double which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public double evaluateDouble() throws EvaluationException {
        return adf[0].evaluateDouble(this);
    }

    /**
     * Evaluate this Individual as a program returning a char.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  A char which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public char evaluateChar() throws EvaluationException {
        return adf[0].evaluateChar(this);
    }

    /**
     * Evaluate this Individual as a program returning a boolean.
     * An Individual is evaluated by evaluating the root (or result producing
     * Chromosome) which is alway assumed to adf[0].
     *
     * @return  A boolean which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public boolean evaluateBoolean() throws EvaluationException {
        return adf[0].evaluateBoolean(this);
    }

    /**
     * Get the total number of Genes in this Individual.
     *
     * @return  the number Genes in the Individual.
     */
    public int complexity() {
        return complexity;
    }

    /**
     * Create a String representing this Individual.
     *
     * @return  A String representing this Individual.
     *
     */
    public String toString() {
        String s = new String("");
        for (int i = 0; i < adf.length; i++)
            s += "ADF[" + i + "] : " + adf[i] + "\n";
        s += "Fitness : " + fitness + "\n";
        s += "Complexity : " + complexity + "\n";
        return s;
    }
}
