/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

/**
 * A GeneTerminal is a leaf in a Gene tree and represents a Terminal.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class GeneTerminal extends Gene implements Cloneable {

    /**
     * Constructs a Gene Terminal by randomly selecting a Terminal of the
     * required type from the Terminal set of the associated Chromosome.
     *
     * @param   maxDepth    the maximum depth of the Gene tree to be generated.
     * @param   type        the required return type of top node in the tree.
     * @param   gpParameters    the GP parameters for this run.
     * @param   adfIndex    the index of the ChromosomeParameters to used.
     * @exception   TypeException   If no Terminal of the required type could
     *                              be found.
     */
    GeneTerminal(int maxDepth, Type type,
        GPParameters gpParameters, int adfIndex) throws TypeException {
        // get an array of all terminals of the required type
        Terminal[] terminalsOfType =
            gpParameters.adf[adfIndex].terminalsOfType.get(type);

        // if no terminals of required type throw a wobbly
        if (terminalsOfType == null)
            throw new TypeException(
                "no Terminal of type " + type.toString() +
                " at depth " + maxDepth);

        // now select one at random from the array and set p to it.
        int code =  gpParameters.rng.nextInt() % terminalsOfType.length;
        code = (code < 0) ? -code : code;
        p = terminalsOfType[code].instance();
    }

    /**
     * Evaluate this Gene as Terminal returning an Object reference.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  An Object which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public Object evaluateObject(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateObject(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a byte.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A byte which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public byte evaluateByte(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateByte(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a short.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A short which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public short evaluateShort(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateShort(i);
    }

    /**
     * Evaluate this Gene as Terminal returning an int.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  An int which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public int evaluateInt(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateInt(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a long.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A long which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public long evaluateLong(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateLong(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a float.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A float which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public float evaluateFloat(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateFloat(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a double.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A double which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public double evaluateDouble(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateDouble(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a char.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A char which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public char evaluateChar(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateChar(i);
    }

    /**
     * Evaluate this Gene as Terminal returning a boolean.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A boolean which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public boolean evaluateBoolean(Individual i) throws EvaluationException {
        return ((Terminal) p).evaluateBoolean(i);
    }

    /**
     * Calculate the maximum depth of this Gene.  The result is always one
     * since this Gene represents a Terminal.
     *
     * @return  1, this the depth of any Terminal.
     *
     */
    public int depth() {
        return 1;
    }

    /**
     * Get the number of Genes in this Gene.
     *
     * @return  1, since it represents a Terminal.
     */
    public int complexity() {
        return 1;
    }


    /**
     * Make a clone of this Gene.
     *
     * @return  a clone of this Gene.
     */
    public Gene deepClone() {
        GeneTerminal clone = null;
        try {
            clone = (GeneTerminal) this.clone();
        }
        catch (CloneNotSupportedException e) {
            // will never happen as long as we implement Cloneable
        }
        return clone;
    }

    /**
     * Create a String representing this Gene.
     *
     * @return  a String representing this tree.
     */
    public String toString() {
        return p.toString();
    }
}
