/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

/**
 * A GeneFunction is a Gene tree representing a function call.  A GeneFunction
 * therefore has Gene arguments (branches or subtrees).
 *
 * @see gpsys.Gene
 * @see gpsys.GeneFunctionGrow
 * @see gpsys.GeneFunctionFull
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public abstract class GeneFunction extends Gene implements Cloneable {
    /**
     * The arguments for this function call.  The length of this array is
     * equal to the number of arguments taken by the function referenced by
     * this Gene.
     */
    public Gene[] arguments;

    /**
     * Evaluate this Gene as Function returning an Object reference.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  An Object which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public Object evaluateObject(Individual i) throws EvaluationException {
        return ((Function) p).evaluateObject(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a byte.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A byte which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public byte evaluateByte(Individual i) throws EvaluationException {
        return ((Function) p).evaluateByte(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a short.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A short which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public short evaluateShort(Individual i) throws EvaluationException {
        return ((Function) p).evaluateShort(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning an int.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  An int which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public int evaluateInt(Individual i) throws EvaluationException {
        return ((Function) p).evaluateInt(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a long.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A long which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public long evaluateLong(Individual i) throws EvaluationException {
        return ((Function) p).evaluateLong(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a float.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A float which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public float evaluateFloat(Individual i) throws EvaluationException {
        return ((Function) p).evaluateFloat(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a double.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A double which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public double evaluateDouble(Individual i) throws EvaluationException {
        return ((Function) p).evaluateDouble(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a char.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A char which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public char evaluateChar(Individual i) throws EvaluationException {
        return ((Function) p).evaluateChar(i, arguments);
    }

    /**
     * Evaluate this Gene as Function returning a boolean.
     *
     * @param   i   the individual to which this Gene belongs
     * @return  A boolean which is the result of the evaluation.
     * @exception   EvaluationException If there is an evaluation failure.
     *
     */
    public boolean evaluateBoolean(Individual i) throws EvaluationException {
        return ((Function) p).evaluateBoolean(i, arguments);
    }

    /**
     * Get the maximum depth of this Gene tree
     *
     * @return  the maximum depth of the tree.
     *
     */
    public int depth() {

        // the depth is initially 0
        int d = 0;

        // get the maximum depth of each branch
        for (int i = 0; i < arguments.length; i++) {
            int tmp = arguments[i].depth();
            d = (d < tmp) ? tmp : d;
        }

        // the maximum depth of this Gene is the 1 + the maximum depth of all
        // the branches
        return d + 1;
    }

    /**
     * Get the number of Genes in this Gene tree.
     *
     * @return  the number of Genes in this Gene tree.
     */
    public int complexity() {

        // the total number of Genes is initially one.
        int sum = 1;

        // add the total number of Genes in each argument
        for (int i = 0; i < arguments.length; i++) {
            sum += arguments[i].complexity();
        }
        // the total is this sum
        return sum;
    }

    /**
     * Make a clone of this Gene tree, cloning all subtrees.
     *
     * @return  a clone of this Gene.
     */
    public Gene deepClone() {
        // the clone is initially empty
        GeneFunction clone = null;
        try {
            // now clone the Gene, and then make arguments refer to clones too
            clone = (GeneFunction) this.clone();
            clone.p = p.instance();
            clone.arguments = new Gene[arguments.length];
            for (int i = 0; i < arguments.length; i++)
                clone.arguments[i] = (Gene) arguments[i].deepClone();
        }
        catch (CloneNotSupportedException e) {
            // will never happen as long as we implement Cloneable
            // which we do !!
        }
        return clone;
    }

    /**
     * Create a String representing this Gene tree.
     *
     * @return  a String representing this tree.
     */
    public String toString() {
        String s = new String();
        s += "(" + p;
        for (int i = 0; i < arguments.length; i++)
            s += " " + arguments[i];
        s += ")";
        return s;
    }
}
