/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

import java.util.Random;
import java.util.Vector;

/**
 * A GeneBranch defines a Gene subtree of a GP Gene tree.  It has
 * instance variables defining the parent Gene of this subtree, and where it
 * is referenced in the parent.  The GeneBranch provides suffiecient
 * information to allow a subtree to be replaced by another one.  The latter
 * could be generated either by creation (for mutation) or crossover.
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class GeneBranch {
    /**
     * The parent of the Gene subtree.
     */
    public Gene parent;

    /**
     * The arguments index of this Gene subtree in the parent Gene.
     */
    public int  index;

    /**
     * The Gene subtree itself.
     */
    public Gene child;

    /**
     * The default size of the vector used for keeping subtrees of a given type.
     */
    public static final int TYPE_VECTOR_SIZE = 10;

    /**
     * Total number of times a function point was selected for mutation or
     * crossover.
     */
    public static long functionCount;

    /**
     * Total number of times a terminal point was selected for mutation or
     * crossover.
     */
    public static long terminalCount;

    /**
     * Construct a GeneBranch using the supplied arguments to initialise
     * the instance variables.
     *
     * @param   parent  the parent of this subtree.
     * @param   index   the arguments index of this Gene subtree in the
     *                  parent Gene.
     * @param   child   the subtree itself.
     */
    private GeneBranch(Gene parent, int index, Gene child) {
        this.parent = parent;
        this.index  = index;
        this.child  = child;
    }

    /**
     * select a subtree at random from the given tree.
     *
     * @param   rng     the random number generator to be used.
     * @param   treeTop the tree from which to select a subtree.
     */
    public GeneBranch(Random rng, Gene treeTop) {
        parent = null;
        selectBranch(rng, treeTop, null);
    }

    /**
     * select a subtree of a given type at random from the given tree.
     *
     * @param   rng     the random number generator to be used.
     * @param   treeTop the tree from which to select a subtree.
     * @param   type    the type of the subtree to be selected.
     */
    public GeneBranch(Random rng, Gene treeTop, Type type) {
        parent = null;
        selectBranch(rng, treeTop, type);
    }

    /**
     * select a subtree of at random from the given tree.  Initialise the
     * instance variable to reflect the selected tree.
     *
     * @param   rng     the random number generator to be used.
     * @param   tree    the tree from which to select a subtree.
     */
    public void selectBranch(Random rng, Gene tree, Type type) {

        // get a table of all the branches of a given type from the tree
        Vector terminalBranches =
            new Vector(TYPE_VECTOR_SIZE, TYPE_VECTOR_SIZE);
        Vector functionBranches =
            new Vector(TYPE_VECTOR_SIZE, TYPE_VECTOR_SIZE);
        branchesOfType(null, -1, tree, type, terminalBranches, functionBranches);

        // if the table is empty, i.e. if there are no branches of this type
        // initialise the instance variables to nulls
        if (terminalBranches.size() == 0 && functionBranches.size() == 0) {
            parent = child = null;
            index = -1;
            return;
        }

        // select a branch at random from the table of branches

        // first decide wether to pick a function or a terminal
        // we a have bias of 90% towards picking a function
        boolean selectFunction = rng.nextInt(100) < 90;

        // if we are to select a function, but there are no function branches to
        // choose from then select a terminal instead.
        if (selectFunction && functionBranches.size() == 0)
            selectFunction = false;

        // if we are to select a terminal, but there are no terminal branches to
        // choose from then select a function instead.
        if (!selectFunction && terminalBranches.size() == 0)
            selectFunction = true;

        if (selectFunction)
            functionCount++;
        else
            terminalCount++;

        int branchIndex;
        GeneBranch branch;
        if (selectFunction) {
            branchIndex = rng.nextInt(functionBranches.size());
            branch = (GeneBranch) functionBranches.elementAt(branchIndex);
        } else { // select a terminal branch
            branchIndex = rng.nextInt(terminalBranches.size());
            branch = (GeneBranch) terminalBranches.elementAt(branchIndex);
        }

        // initialise the instance variables to this branch
        parent  = branch.parent;
        child   = branch.child;
        index   = branch.index;
    }

    /**
     * Get a table of all subtrees of a given type.
     *
     * @param   parent      the parent Gene of the child.
     * @param   index       the arguments index of the  child in the parent.
     * @param   child       the child Gene subtree.
     * @param   type        the type of subtree required (null if don't care).
     * @param   terminalBranches    the table of subtrees that are terminals.
     * @param   functionBranches    the table of subtrees that are functions.
     */
    // return a vector of all geneBranches of a type starting at child
    public static void branchesOfType(Gene parent, int index, Gene child,
        Type type, Vector terminalBranches, Vector functionBranches) {

        // if the child has the same type as required or no type
        // is required add the child to the vector
        if (type == null || child.p.type == type) {
            if (child instanceof GeneFunction)
                terminalBranches.addElement(new GeneBranch(parent, index, child));
            else // child instanceof GeneTerminal
                functionBranches.addElement(new GeneBranch(parent, index, child));
        }

        // if the child is a function, recurse over each argument
        // the parent for these recursive calls is the current child
        if (child instanceof GeneFunction)
            for (int i = 0; i < ((GeneFunction) child).arguments.length; i++)
                branchesOfType(child, i, ((GeneFunction) child).arguments[i],
                    type, terminalBranches, functionBranches);
    }
}
