/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

import java.util.Date;
import java.util.Random;
import java.io.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * <p>
 * This class is used to parameterise the GP system.  Use this class as is
 * or extend it for your problem, but in either case fill in the instance
 * variables to specify your problem.
 * </p>
 *
 * <p>
 * The class supports methods enabling loading and saving of a GPParameters
 * object.  The object may be written/read to/from either a file or a stream.
 * These methods use Java Serialisable Objects and hence if you extend the
 * class to include new instance variables, ensure that they are Serializable
 * so that they can be saved.  If any instance variables you introduce
 * are temporary variables, declare them transient to ensure they are not
 * saved.
 * </p>
 *
 * <p>
 * There are also methods enabling a generation report to be sent to a file
 * or an OutputStream.
 * </p>
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class GPParameters implements Serializable {
    /**
     * The object monitoring the GP system.
     */
    public transient GPObserver observer;

    /**
     * The engine to be used for evolution.  The value of this variable must
     * be set to one of the class constants beggining with ENGINE_ such as
     * ENGINE_STEADYSTATE.
     */
    public int          engine;

    /**
     * Used for selecting the steady state engine type.  A Steady State engine
     * is useful if you are low on memory.  Each new individual created via
     * mutation or xover immediately replaces an individual in the population
     * and is hence immediately available for creating the next individual.
     * The concept of a generation is when an entire population of new
     * Individuals have been created.  In a Steady State engine the best
     * individual of the generation is also the best individual of the run.
     */
    public static final int ENGINE_STEADYSTATE  = 0;

    /**
     * Used for selecting the generational engine type.  A Generational engine
     * is useful if the application requires that each generation is given a
     * different set of test cases for fitness evaluation.  A Generational
     * engine creates an entire new population using xover and mutation which
     * replaces the old population.  In a Generational engine the best
     * individual of generation need not be the best of the run.
     */
    public static final int ENGINE_GENERATIONAL = 1;

    /**
     * The random number generator seed.
     */
    public long         rngSeed;

    /**
     * The random number generator.  This should have been created with the
     * above seed.
     */
    public Random       rng;

    /**
     * The probability of mutation.  This needs to be a number between 0 and 1.
     */
    public double       pMutation;

    /**
     * The probability of Reproduction.  This needs to be a number between 0 and
     * 1.
     */
    public double       pReproduction;

    /**
     * The tournament size to be used for tournament selection during crossover
     * and mutation.
     */
    public int          tournamentSize;

    /**
     * How many Individuals there are in the population.
     */
    public int          populationSize;

    /**
     * The number of generations to evolve.
     */
    public int          generations;

    /**
     * Parameters for each ADF to be evolved.  The length of this array is
     * effectively the number of ADFs required.
     */
    public ChromosomeParameters[] adf;

    /**
     * The Fitness definition for the problem.
     */
    public Fitness      fitness;

    /**
     * The population.  This field is filled in automatically by the GP system.
     */
    public Population   population;

    /**
     * Used during the generation of the initial popupation.  It is the index
     * of the current Individual being created.
     */
    protected int       creationIndex;

    /**
     * Save this object to a file compressing it using gzip during the save.
     * This method is typically called by GPObservers after every generation
     * to enable restarting from the last saved point in the event of system
     * failure.
     *
     * @param   filePrefix  the file prefix to be used for save files.  Two
                            files may be created :-
                                filePrefix.p1.gzip and
                                filePrefix.p2.gzip
     */
    public void save(String filePrefix) throws IOException {

        File f1 = new File(filePrefix + ".p1.gzip");
        File f2 = new File(filePrefix + ".p2.gzip");

        // indicates an aborted save during the last execution of the GP system
        // f2 is probably corrupt, so throw it away.
        if (f2.exists())
            f2.delete();

        // we try to save to f2 first then move f2 to f1, that way if anything
        // goes wrong, we will still have f1.
        FileOutputStream fos = new FileOutputStream(f2);
        save(fos);
        fos.close();

        // to get here means that we were succesful, so get rid of f1 if it
        // exist and rename f2 to f1.
        f1.delete();
        f2.renameTo(f1);
    }

    /**
     * Write this object to an OutputStream compressing it using gzip as it
     * is written.
     *
     * @param   os  the output stream to be written to.
     */
    public void save(OutputStream os) throws IOException {
        GZIPOutputStream    zos = new GZIPOutputStream(os);
        ObjectOutputStream  oos = new ObjectOutputStream(zos);

        oos.writeObject(this);

        oos.close();
        zos.close();
    }

    /**
     * Loads a previously saved GPParameters object from the file specified.
     * The GPParameters object must previously have been saved by the save()
     * method.  The save file is assumed to be gzipped and hence is
     * decompressed during the read.
     *
     * @param   filePrefix  the file prefix used by the save files.
     * @return  the GPParameters object read from the file.
     * @exception ClassNotFoundException
     *              if the class definition for one of the objects being read
     *              cannot be found.
     */
    public static GPParameters load(String filePrefix)
    throws IOException, ClassNotFoundException {

        File f1 = new File(filePrefix + ".p1.gzip");
        File f2 = new File(filePrefix + ".p2.gzip");

        // Ensure save file is there
        if (!f1.exists())
            if (f2.exists())        // => crashed b4 f2 renamed to f1 in save()
                f2.renameTo(f1);    // so rename it !!

        if (f1.canRead() && f1.canWrite()) {

            // open file f1 and load the GPParameters object from it
            FileInputStream fis = new FileInputStream(f1);
            GPParameters gpParameters = load(fis);
            fis.close();

            return gpParameters;
        }
        return null;
    }

    /**
     * Load a GPParameters object from the InputStream.  The stream is assumed
     * to be gzipped and is hence unzipped during the load.
     *
     * @param   is  the input stream to read from.
     * @return  the GPParameters object read from the stream.
     * @exception ClassNotFoundException
     *              if the class definition for one of the objects being read
     *              cannot be found.
     */
    public static GPParameters load(InputStream is)
    throws IOException, ClassNotFoundException {

        GZIPInputStream     zis = new GZIPInputStream(is);
        ObjectInputStream   ois = new ObjectInputStream(zis);

        GPParameters gpParameters = (GPParameters) ois.readObject();

        ois.close();
        zis.close();

        return gpParameters;
    }

    /**
     * Write a report of the current generation to the PrintWriter.
     * The first report also contains the random number generator seed,
     * the population size and the number of generations to be evolved.
     *
     * @param   pw          the PrintWriter to use to print the report.
     * @param   firstReport specifies whether this the first report. 
     */
    public void writeReport(PrintWriter pw, boolean firstReport) {
        if (firstReport) {
            pw.println("RNG seed    = " + rngSeed);
            pw.println("Population  = " + populationSize);
            pw.println("Generations = " + generations);
        }
        population.report(pw);
    }

    /**
     * Write a report of the current generation to a file.
     * The first report also contains the random number generator seed,
     * the population size and the number of generations to be evolved.
     *
     * @param   filePrefix  the filename prefix to use for the file to which
                            the report is written.  The file created is named :
                                filePrefix.txt
     * @param   firstReport specifies whether this the first report. 
     */
    public void writeReport(String filePrefix, boolean firstReport)
    throws IOException {
        FileOutputStream fos = new FileOutputStream(filePrefix + ".txt", true);
        PrintWriter pw = new PrintWriter(fos);

        writeReport(pw, firstReport);

        pw.close();
        fos.close();
    }
}
