/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

/**
 * This is an implementation of the bookkeeping class used for memory efficient
 * crossover as described in the book Genetic Programming III.
 *
 * @see gpsys.Population
 * @see gpsys.GPParameters
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
class CrossoverBookkeeping {
    /**
     * The list heads in this array are used to create 4 lists.  These lists
     * hold information about parents with 0, 1, 2 or more operations respectively.
     */
    ParentInfo[] listHead;
    /**
     * The parent table is 2 elements bigger than the number of individuals in
     * the population.  It holds information about each individual as a
     * potential parent.  The extra 2 slots are required by the algorithm to
     * gurantee that sufficient space is avaliable for newly created
     * individuals.
     */
    ParentInfo[] parentTable;

    /**
     * Construct a bookkeeping object suitable for storing the selected genetic
     * operations for creating the next generation.
     *
     * @param size  The size of the population.
     */
    CrossoverBookkeeping(int size) {
        parentTable = new ParentInfo[size + 2];
        for (int i = 0; i < parentTable.length; i ++) {
            parentTable[i] = new ParentInfo(i);
        }

        listHead = new ParentInfo[4];
        for (int i = 0; i < listHead.length; i++)
            listHead[i] = new ParentInfo(-1);
    }


    /**
     * Add a crossover operation to the list of operations to be performed.
     *
     * @param p1    The index of the first parent.
     * @param p1    The index of the second parent.
     */
    void addCrossoverOperation(int p1, int p2) {
        ParentInfo parent1 = parentTable[p1];
        ParentInfo parent2 = parentTable[p2];
        parent1.addCrossover(parent2);
        parent2.addCrossover(parent1);
    }
        
    /**
     * Add a reproduction operation to the list of operations to be performed.
     *
     * @param p    The index of the individual to be reproduced.
     */
    void addReproductionOperation(int p) {
        ParentInfo parent = parentTable[p];
        parent.addReproduction();
    }
    
    /**
     * Add a mutation operation to the list of operations to be performed.
     *
     * @param p     The index of the individual to be mutated.
     */
    void addMutationOperation(int p) {
        ParentInfo parent = parentTable[p];
        parent.addMutation();
    }

    /**
     * Remove a genetic operation from the list of operations to be performed.  The
     * selected operation is guranteed to involve a parent with the least number of
     * operations remaining.
     *
     * @return  The next GeneticOperation to be performed.
     */
    GeneticOperation removeOperation() {
        // find the first list which is not empty.  The lists are ordered so that
        // lists earlier in the array have parents with fewer remaining operations.
        int i = 1;
        while (listHead[i].next == null)
            i++;
        // remove the first parent from the list
        ParentInfo p = listHead[i].next;
        // remove an operation from the parent
        GeneticOperation op = p.removeOperation();
        // now reposition the parent in the right list
        removeFromList(parentTable[op.parent1]);
        addToList(parentTable[op.parent1]);
        // if the operation was a crossover, better do the same for the second parent.
        if (op instanceof CrossoverOperation) {
            CrossoverOperation cx = (CrossoverOperation) op;
            // remove the crossover operation involving  the first parent.
            parentTable[cx.parent2].removeCrossover(parentTable[cx.parent1]);
            // and reposition the second parent in the right list
            removeFromList(parentTable[cx.parent2]);
            addToList(parentTable[cx.parent2]);
        }
        return op;
    }

    /**
     * Remove a parent from the free list.
     *
     * @return  the index of the removed parent.
     */
    int removeFreeSlot() {
        // get the first parent in the free list
        ParentInfo p = listHead[0].next;
        // remove the parent from the list and kill it
        removeFromList(p);
        p.kill();
        // return the population index of the parent
        return p.id;
    }

    /**
     * Populate the four lists with parent mating information.
     */
    void createLists() {
        // place the parent information structures into the appropriate lists depending
        // on their operation counts
        for (int i = 0; i < parentTable.length; i++)
            addToList(parentTable[i]);
    }

    /**
     * Add the parent mating information structure to the appropriate list.
     * The parent information will be added to either the 1st, 2nd, 3rd or 4rth list
     * depending on wether the parent is ivolved in 0, 1, 2 or more than two genetic
     * operations respectively.
     */
    private void addToList(ParentInfo parent) {
        // first get their operation counts
        int opCount = parent.getOperations();
        // use the opCount to determine which list to add the parent into
        opCount = (opCount <= 2) ? opCount : 3;
        // prepend the parent to that list
        parent.prev = listHead[opCount];
        parent.next = listHead[opCount].next;
        if (parent.next != null)
            parent.next.prev = parent;
        listHead[opCount].next = parent;
    }

    /**
     * Remove the parent mating information structure from the associated list.
     */
    private void removeFromList(ParentInfo parent) {
        // remove the parent from the it's associated list
        // link the item above the parent to the next item
        parent.prev.next = parent.next;
        // link the item below the parent to the previous item
        if (parent.next != null)
            parent.next.prev = parent.prev;
        parent.prev = null;
        parent.next = null;
    }

    /**
     * Reset the bookkeeping data structure.  This method resets each parent
     * information data structure and ensures that the 4 lists are empty.
     */
    void reset() {
        // reset each parent information block
        for (int i = 0; i < parentTable.length; i++)
            parentTable[i].reset();

        // reset the list heads
        for (int i = 0; i < listHead.length; i++)
            listHead[i].next = null;
    }

    /**
     * Create a string representation of the bookkeeping data structure.
     *
     * @return  A String representing the current state of this object.
     */
    public String toString() {
        String s = "";

        for (int i = 0; i < listHead.length; i++) {
            s += "list[" + i + "] = \n";
            for (ParentInfo p = listHead[i].next; p != null; p = p.next)
                s += p + "\n";
        }
        return s;
    }
}
