/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

/**
 * ChromosomeParameters define information needed to construct a chromosome.
 * Users should extend this class for each ADF to be defined.
 *
 * @see gpsys.Chromsome
 * @see gpsys.GPParameters
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class ChromosomeParameters implements java.io.Serializable {

    /**
     * The type to be returned by the Chromosome.
     *
     * @see Type
     */
    public Type type;

    /**
     * The maximum depth of the Gene tree.
     */
    public int  maxDepth;

    /**
     * The maximum depth at creation of the Gene tree.
     */
    public int  maxDepthAtCreation; 

    /**
     * The maximum depth of tree created to mutate the Gene tree.
     */
    public int  maxDepthMutation;

    /**
     * The table of available functions.
     */
    public Function[] functions;

    /**
     * The table of available terminals.
     */
    public Terminal[] terminals;

    /**
     * The table of available types.
     */
    public Type[]   types;

    /**
     * Functions of a given return type possible at a given depth
     * for both the GROW creation method.
     * The index = the required depth, the array contains a hash table mapping
     * Types to Function arrays - note that not all functions of given return
     * type possible at a given depth.
     */
    public TypeToFunctionsTable[] functionsOfTypeAtDepthGrow;

    /**
     * Functions of a given return type possible at a given depth
     * for both the FULL creation method.
     * The index = the required depth, the array contains a hash table mapping
     * Types to Function arrays - note that not all functions of given return
     * type possible at a given depth.
     */
    public TypeToFunctionsTable[] functionsOfTypeAtDepthFull;

    /**
     * A hash table mapping a given Type to an array of terminals of that
     * Type.
     */
    public TypeToTerminalsTable terminalsOfType;

    /**
     * The method of Chromosome creation, this can be initialised to any
     * of the following constants.
     */
    public int createMethod;

    /**
     * This constant used to initialise the createMethod variable to
     * select the full method of creation.  The full method creates Gene trees
     * of the maximum available depth.
     */
    public static final int CREATE_FULL = 0;

    /**
     * This constant used to initialise the createMethod variable to
     * select the grow method of creation.  The grow method creates Gene trees
     * of variable depths.
     */
    public static final int CREATE_GROW = 1;

    /**
     * This constant used to initialise the createMethod variable to
     * select the ramp method of creation.  The ramp half and half method
     * creates Gene trees in the population of which half are created
     * using the full method and the other half are created using the
     * grow method.  The depth of these trees is varied between 2 and
     * maxDepthAtCreation.
     */
    public static final int CREATE_RAMP_HALF_AND_HALF = 2;

    /**
     * Initialises the type possibilities tables, read the
     * "Strongly Typed Genetic Programing" paper by David J. Montana.
     *
     * @param   maxDepthAtCreation  the maximum depth at creation of the
     *          Gene trees.
     */
    public void initialiseTypeTables(int maxDepthAtCreation) {
        initialiseTerminalsTypeTable();
        initialiseFunctionsTypeTable(maxDepthAtCreation);

    }

    private void initialiseTerminalsTypeTable() {
        // setup the terminals type table
        // might not have terminals for all possible types, but so what !!!
        terminalsOfType = new TypeToTerminalsTable(types.length);
        for (int t = 0; t < types.length; t++) {

            // assume all the terminals are of this type
            Terminal[] tmp = new Terminal[terminals.length];

            // now add ones that really are of this type while keeping count
            int count = 0;
            for (int i = 0; i < terminals.length; i++)
                if (terminals[i].type == types[t]) {
                    tmp[count] = terminals[i];
                    count++;
                }
            if (count == 0)
                tmp = null;
            else {
                // now store terminals of this type in an array of exact size
                Terminal[] tmp1 = tmp;
                tmp = new Terminal[count];
                for (int i = 0; i < count; i++)
                    tmp[i] = tmp1[i];
                terminalsOfType.put(types[t], tmp);
            }
        }
    }

    private void initialiseFunctionsTypeTable(int maxDepthAtCreation) {
        // setup the functions tables, much harder than terminals !!!

        initialiseFunctionsTypeTableFull(maxDepthAtCreation);
        initialiseFunctionsTypeTableGrow(maxDepthAtCreation);
    }

     private void initialiseFunctionsTypeTableFull(int maxDepthAtCreation) {

        functionsOfTypeAtDepthFull =
            new TypeToFunctionsTable[maxDepthAtCreation];

        for (int d = 1 ; d < functionsOfTypeAtDepthFull.length; d++) {

            // might not have functions of all types  at this depth, but so what
            functionsOfTypeAtDepthFull[d] =
                new TypeToFunctionsTable(types.length);

            for (int t = 0; t < types.length; t++) {

                // assume all functions are of this type & available at depth
                Function[] tmp = new Function[functions.length];

                // now add the functions to the array
                int countFull = 0;
                functionsLoop:
                for (int f = 0; f < functions.length; f++)
                    // the return type must be what we expect
                    if (functions[f].type == types[t]) {
                        // and the argument types must be available
                        for (int a = 0; a < functions[f].argTypes.length; a++)
                            if (d == 1) {
                                // there are no terminals of the same type as
                                // this argument, so skip this function
                                if (terminalsOfType.
                                    get(functions[f].argTypes[a]) == null)
                                    continue functionsLoop;
                            } else {
                                // there are no functions of the same type as
                                // this argument in the previous depth
                                if ((d == 1) ||
                                    (functionsOfTypeAtDepthFull[d - 1].
                                    get(functions[f].argTypes[a]) == null))
                                    continue functionsLoop;
                            }

                        tmp[countFull] = functions[f];
                        countFull++;
                    }

                if (countFull == 0)
                    tmp = null;
                else {
                    // now create a packed array of the functions
                    Function[] tmp1 = tmp;
                    tmp = new Function[countFull];
                    for (int f = 0; f < countFull; f++)
                        tmp[f] = tmp1[f];
                    // now insert the array into the hash table for this type
                    functionsOfTypeAtDepthFull[d].put(types[t], tmp);
                }
            }
        }
    }


    private void initialiseFunctionsTypeTableGrow(int maxDepthAtCreation) {

        functionsOfTypeAtDepthGrow =
            new TypeToFunctionsTable[maxDepthAtCreation];

        for (int d = 1 ; d < functionsOfTypeAtDepthGrow.length; d++) {

            // might not have functions of all types  at this depth, but so what
            functionsOfTypeAtDepthGrow[d] =
                new TypeToFunctionsTable(types.length);

            for (int t = 0; t < types.length; t++) {

                // assume all functions are of this type & available at depth
                Function[] tmp = new
                    Function[functions.length];

                // now add the functions to the array
                int countGrow = 0;
                if (d > 1) {
                    // for the Grow method need to add all functions appearing
                    // in table entry for this type at depth - 1
                    Function[] functionsBelow =
                        functionsOfTypeAtDepthGrow[d - 1].get(types[t]);
                    if (functionsBelow != null)
                        for (; countGrow < functionsBelow.length; countGrow++)
                            tmp[countGrow] = functionsBelow[countGrow];
                }

                functionsLoop:
                for (int f = 0; f < functions.length; f++)
                    // the return type must be what we expect
                    if (functions[f].type == types[t]) {
                        // now check if the argument types are available
                        for (int a = 0; a < functions[f].argTypes.length; a++)
                            // there are no terminals of the same type as
                            // this argument, and there are no functions of
                            // the same type as so skip this function
                            // this argument in the previous depth
                            // hence this function not possibe at this depth
                            if (terminalsOfType.get(functions[f].argTypes[a])
                                == null)
                                if ((d == 1) ||
                                    (functionsOfTypeAtDepthFull[d - 1].
                                    get(functions[f].argTypes[a]) == null))
                                    continue functionsLoop;

                        // find out if this function is already in our table
                        int f1;
                        for (f1 = 0; f1 < countGrow; f1++)
                            if (tmp[f1] == functions[f])
                                break;

                        // if f is not already in the table, insert it
                        if (f1 == countGrow) {
                            tmp[countGrow] = functions[f];
                            countGrow++;
                        }
                    }

                if (countGrow == 0)
                    tmp = null;
                else {
                    // now create a packed array of the functions
                    Function[] tmp1 = tmp;
                    tmp = new Function[countGrow];
                    for (int f = 0; f < countGrow; f++)
                        tmp[f] = tmp1[f];
                    // now insert the array into the hash table for this type
                    functionsOfTypeAtDepthGrow[d].put(types[t], tmp);
                }
            }
        }
    }
}
