/*
    Copyright Adil Qureshi - $Date: 2000/07/05 16:21:25 $
    This code is part of gpsys Release $Name: 2b $
    and is released for non-commercial use only.
    Questions, comments etc should be forwarded to :-

        Adil Qureshi
        University College London,
        Department of Computer Science,
        Gower St,
        London WC1E 6BT, UK.
        email: A.Qureshi@cs.ucl.ac.uk
        URL : http://www.cs.ucl.ac.uk/staff/A.Qureshi/
*/

package gpsys;

/**
 * A Chromosome defines an evolvable gene tree.
 *
 * @see gpsys.Terminal
 *
 * @version     $Revision: 1.1 $, $Date: 2000/07/05 16:21:25 $
 * @author  <a href="mailto:A.Qureshi@cs.ucl.ac.uk">Adil Qureshi</a>
 *          <address>Department of Computer Science,</address>
 *          <address>University College London,</address>
 *          <address>Gower St,</address>
 *          <address>London WC1E 6BT,</address>
 *          <address>UK.</address>
 */
public class Chromosome implements Cloneable, java.io.Serializable  {

    /**
     * The Gene at the top of the tree.
     * this gene is evaulated when this ADF needs evaluation
     *
     * @see Gene
     */
    public Gene treeTop;

    /**
     * The GPParameters used to create this chromosome
     *
     * @see GPParameters
     */
    public  GPParameters gpParameters;

    /**
     * index into the adf array in the gpParameters.  It is used to acces the
     * ChromosomeParameters associated with this chromosome.
     */
    int adf;

    /**
     * A count of the total number of nodes in this tree.
     */
    int complexity;

    /**
     * Constructs a new Chromosome using the specified GPParameters.
     *
     * @param   p           the GPParameters to use.
     * @param   adfIndex    the index into the adf array in GPParameters
     *                      the latter defines ChromosomeParameters for each
     *                      adf.
     * @exception   TypeException   If there was a typing problem during tree
     *                              generation.  For example a Function or
     *                              Terminal of the required type could not be
     *                              found.
     */
    Chromosome(GPParameters p, int adfIndex) throws TypeException {
        gpParameters = p;
        adf = adfIndex;

        // always start the tree with a function

        int createMethod = p.adf[adfIndex].createMethod;
        if (createMethod == ChromosomeParameters.CREATE_GROW)
            treeTop = new
                GeneFunctionGrow(p.adf[adfIndex].maxDepthAtCreation - 1,
                    p.adf[adfIndex].type, p, adfIndex);
        else if (createMethod == ChromosomeParameters.CREATE_FULL)
            treeTop = new
                GeneFunctionFull(p.adf[adfIndex].maxDepthAtCreation - 1,
                    p.adf[adfIndex].type, p, adfIndex);
        else { // assume (createMethod == CREATE_RAMP_HALF_AND_HALF)

            // ramp up the depth; minimum = 2, maximum = maxDepthAtCreation
            double depthValue = p.adf[adfIndex].maxDepthAtCreation *
                ((float)p.creationIndex / (float)p.populationSize);
            int maxDepth = (depthValue < 2) ? 2 : (int) depthValue;

            // half the population created via grow, and the other via full
            if ((p.creationIndex % 2) == 0)
                treeTop = new
                    GeneFunctionGrow(maxDepth,
                        p.adf[adfIndex].type, p, adfIndex);
            else
                treeTop = new
                    GeneFunctionFull(maxDepth,
                        p.adf[adfIndex].type, p, adfIndex);
        }

        complexity = treeTop.complexity();
    }

    /**
     * Creates a new child Chromosome which is a mutation of the mother.
     *
     * @param   mum is the mother Chromosome.
     * @return  a reference to a child Chromosome which is a mutation of
     *          the mother
     *
     */
    public static Chromosome mutate(Chromosome mum) {

        // a convenience variable
        GPParameters gpParameters = mum.gpParameters;

        // child is initially a clone of it's mum
        Chromosome child = mum.deepClone();

        // pick a branch at random from the child
        GeneBranch branch = new GeneBranch(gpParameters.rng, child.treeTop);

        // just for convenience
        int depth = gpParameters.adf[child.adf].maxDepthMutation - 1;

        // now generate the mutant branch using FULL or GROW methods
        Gene newBranch;
        if ((gpParameters.rng.nextInt() % 2) == 0) {    // use FULL method
            try {
                newBranch   = new
                    GeneFunctionFull(depth, branch.child.p.type,
                        gpParameters, child.adf);
            }
            catch (TypeException e) {
                newBranch = null;
            }
        } else {                                        // use GROW method
            try {
                newBranch   = new
                    GeneFunctionGrow(depth, branch.child.p.type,
                        gpParameters, child.adf);
            }
            catch (TypeException e) {
                newBranch = null;
            }
        }
        if (newBranch == null) {
            try {
                newBranch   = new
                    GeneTerminal(depth, branch.child.p.type,
                        gpParameters, child.adf);
            }
            catch (TypeException e) {
                return child;
            }
        }

        // if the root of child's tree was mutated, replace the whole tree
        if (branch.parent == null)
            child.treeTop = newBranch;
        else {
            // save the branch to be mutated
            Gene tmp = ((GeneFunction) branch.parent).arguments[branch.index];

            // update the tree to include the mutated branch
            ((GeneFunction) branch.parent).arguments[branch.index] = newBranch;

            // if the mutated tree is too big, child == mum
            if (child.treeTop.depth() > gpParameters.adf[child.adf].maxDepth) {
                ((GeneFunction) branch.parent).arguments[branch.index] = tmp;
                gpParameters.observer.diagnosticUpdate(
                    "Throwing away a tree after mutation");
            }
        }
        child.complexity = child.treeTop.complexity();
        return child;
    }

    /**
     * Creates a new child Chromosome via crossover of the mother and father
     * Chromosomes.
     *
     * @param   mum is the mother Chromosome.  The child is actually a copy
     *          of the mother with one branch exchanged with a branch from
     *          the father.
     * @param   dad is the father Chromosome.
     * @return  a reference to a new child Chromosome.
     */
    public static Chromosome cross(Chromosome mum, Chromosome dad) {

        // a convenience variable
        GPParameters gpParameters = mum.gpParameters;

        if (mum == dad)
            gpParameters.observer.diagnosticUpdate(
                "incestious XOVER");

        // make the child a clone of mum
        Chromosome child = mum.deepClone();

        // pick a brach at random from the child to replace
        GeneBranch branchMum = new GeneBranch(gpParameters.rng, child.treeTop);

        // pick a branch from dad to replace it with
        // the new branch must return the same type as the mum branch
        GeneBranch branchDad = new
            GeneBranch(gpParameters.rng, dad.treeTop, branchMum.child.p.type);

        // if now such branch found in dad, child == mum
        if (branchDad.child == null) {
            gpParameters.observer.diagnosticUpdate(
                "Couldn't find compatible branch in dad during crossover");
            return child;
        }

        // make a copy of dad's branch
        Gene newBranch = branchDad.child.deepClone();

        // if replacing root of the child tree, tree = copy of dad's branch
        if (branchMum.parent == null)
            child.treeTop = newBranch;
        else {
            // save the branch being replaced
            Gene tmp =
                ((GeneFunction) branchMum.parent).arguments[branchMum.index];

            // replace the branch with one from dad
            ((GeneFunction) branchMum.parent).arguments[branchMum.index] =
                newBranch;

            // if the resulting tree is too big, child == mum
            if (child.treeTop.depth() > gpParameters.adf[child.adf].maxDepth) {
                ((GeneFunction) branchMum.parent).arguments[branchMum.index] =
                    tmp;
                gpParameters.observer.diagnosticUpdate(
                    "Throwing away a tree after Xover");
            }
        }
        child.complexity = child.treeTop.complexity();
        return child;
    }

    /**
     * Makes a deep copy of this Chromsome by making a copy of the entire
     * data graph.
     *
     * @return  a reference to a copy of this Chromosome.
     */
    public Chromosome deepClone() {
        Chromosome clone = null;
        try {
            clone = (Chromosome) this.clone();
            clone.treeTop = this.treeTop.deepClone();
        }
        catch (CloneNotSupportedException e) {
            // will never happen as long as we implement Cloneable
        }
        return clone;
    }
    
    /**
     * Calculates the number of nodes in the Gene tree of this Chromosome.
     *
     * @return  the number of nodes in the Gene tree.
     */
    public int complexity() {
        return complexity;
    }

    /**
     * Evaluates the chromosome so that it returns an Object reference.
     *
     * @param   i is the Individual being evaluated.
     * @return  a reference to an Object returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public Object evaluateObject(Individual i) throws EvaluationException {
        return treeTop.evaluateObject(i);
    }

    /**
     * Evaluates the chromosome so that it returns a byte.
     *
     * @param   i is the Individual being evaluated.
     * @return  a byte returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public byte evaluateByte(Individual i) throws EvaluationException {
        return treeTop.evaluateByte(i);
    }

    /**
     * Evaluates the chromosome so that it returns a byte.
     *
     * @param   i is the Individual being evaluated.
     * @return  a byte returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public short evaluateShort(Individual i) throws EvaluationException {
        return treeTop.evaluateShort(i);
    }

    /**
     * Evaluates the chromosome so that it returns an int.
     *
     * @param   i   is the Individual being evaluated.
     * @return  an  int returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public int evaluateInt(Individual i) throws EvaluationException {
        return treeTop.evaluateInt(i);
    }

    /**
     * Evaluates the chromosome so that it returns a long.
     *
     * @param   i is the Individual being evaluated.
     * @return  a long returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public long evaluateLong(Individual i) throws EvaluationException {
        return treeTop.evaluateLong(i);
    }

    /**
     * Evaluates the chromosome so that it returns a float.
     *
     * @param   i is the Individual being evaluated.
     * @return  a float returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public float evaluateFloat(Individual i) throws EvaluationException {
        return treeTop.evaluateFloat(i);
    }

    /**
     * Evaluates the chromosome so that it returns a double.
     *
     * @param   i is the Individual being evaluated.
     * @return  a double returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public double evaluateDouble(Individual i) throws EvaluationException {
        return treeTop.evaluateDouble(i);
    }

    /**
     * Evaluates the chromosome so that it returns a char.
     *
     * @param   i is the Individual being evaluated.
     * @return  a char returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public char evaluateChar(Individual i) throws EvaluationException {
        return treeTop.evaluateChar(i);
    }

    /**
     * Evaluates the chromosome so that it returns a boolean.
     *
     * @param   i is the Individual being evaluated.
     * @return  a boolean returned by evaluating the Gene tree.
     * @exception   EvaluationException If there is an evaluation failure.
     */
    public boolean evaluateBoolean(Individual i) throws EvaluationException {
        return treeTop.evaluateBoolean(i);
    }

    /**
     * Generates a String representing  a dump of the Gene tree for this
     * Chromosome.
     *
     * @return  a String representing the Gene tree for this Chromosome.
     */
    public String toString() {
        return treeTop.toString();
    }

}
